#!/bin/bash

################################################################
# Configs
################################################################

NVIDIA_DRIVER_VERSION="570"

################################################################
# Debian Unique Functions
################################################################

debian_write_sources_list()
{
	local deb_release=""

	if [[ -f /etc/os-release ]]; then
		. /etc/os-release
	fi

	deb_release="${VERSION_CODENAME:?}"

	cat > /etc/apt/sources.list <<- EOF
		deb https://deb.debian.org/debian ${deb_release} main contrib non-free non-free-firmware
		deb-src https://deb.debian.org/debian ${deb_release} main contrib non-free non-free-firmware

		deb https://deb.debian.org/debian-security/ ${deb_release}-security main contrib non-free non-free-firmware
		deb-src https://deb.debian.org/debian-security/ ${deb_release}-security main contrib non-free non-free-firmware

		deb https://deb.debian.org/debian ${deb_release}-updates main contrib non-free non-free-firmware
		deb-src https://deb.debian.org/debian ${deb_release}-updates main contrib non-free non-free-firmware

		#deb https://deb.debian.org/debian ${deb_release}-backports main contrib non-free non-free-firmware
		#deb-src https://deb.debian.org/debian ${deb_release}-backports main contrib non-free non-free-firmware
	EOF

	if [[ "${OCTENIUM_COMMON_ARCHITECTURE}" == x86_64 ]]; then
		cat >> /etc/apt/sources.list <<- EOF

			deb https://debian.mirror.constant.com ${deb_release} main contrib non-free non-free-firmware
			deb-src https://debian.mirror.constant.com ${deb_release} main contrib non-free non-free-firmware
		EOF
	fi

	if [[ "${deb_release}" == bullseye ]]; then
		sed -i -e 's/ non-free-firmware//g' /etc/apt/sources.list
	fi
}

################################################################
# Ubuntu Unique Functions
################################################################

ubuntu_install_cockpit()
{
	install_packages \
		cockpit \
		cockpit-bridge \
		cockpit-doc \
		cockpit-networkmanager \
		cockpit-packagekit \
		cockpit-storaged \
		cockpit-system

	initsvc disablenow cockpit.socket

	if grep -qsx root /etc/cockpit/disallowed-users > /dev/null 2>&1; then
		sed -i '/^root$/d' /etc/cockpit/disallowed-users
	fi

if [[ -f /opt/oct-installer/cockpit.conf && -d /etc/nginx/sites-available ]]; then
		mv /opt/oct-installer/cockpit.conf /etc/nginx/sites-available/
	elif [[ -d /opt/oct-installer/cockpit ]]; then
		mv /opt/oct-installer/cockpit/cockpit_nginx.conf /etc/nginx/sites-available/cockpit.conf
		mv /opt/oct-installer/cockpit/*.sh /opt/octenium/
	fi

	[[ -z "${APP_SHORT:-}" ]] || sed -i -e "s/{{APP}}/${APP_SHORT}/g" /opt/octenium/cockpit-certificate.sh
}

ubuntu_configure_cockpit()
{
	local \
		ip="${1:?}" \
		port="${2:-"9080"}"

	sed -i -e "s/{{PORT}}/${port}/g" /etc/nginx/sites-available/cockpit.conf
	ln -s /etc/nginx/sites-available/cockpit.conf /etc/nginx/sites-enabled/cockpit.conf

	mkdir -p /etc/systemd/system/cockpit.socket.d
	cat > /etc/systemd/system/cockpit.socket.d/listen.conf <<- EOF
		[Socket]
		ListenStream=
		ListenStream=1${port}
	EOF

	cat > /etc/cockpit/cockpit.conf <<- EOF
		[WebService]
		Origins = https://${ip}:${port} wss://${ip}:${port}
		ProtocolHeader = X-Forwarded-Proto
		AllowUnencrypted = true
	EOF

	initsvc daemonreload
}

ubuntu_install_php()
{
	local php_vers="${1:?"Specify desired PHP version!"}"

	echo "${php_vers}" > /opt/oct-installer/phpver

	LC_ALL=C.UTF-8 add-apt-repository -y ppa:ondrej/php

	install_packages \
		"php${php_vers}-bcmath" \
		"php${php_vers}-bz2" \
		"php${php_vers}-cli" \
		"php${php_vers}-common" \
		"php${php_vers}-curl" \
		"php${php_vers}-fpm" \
		"php${php_vers}-gd" \
		"php${php_vers}-gmp" \
		"php${php_vers}-igbinary" \
		"php${php_vers}-imagick" \
		"php${php_vers}-intl" \
		"php${php_vers}-mbstring" \
		"php${php_vers}-mysql" \
		"php${php_vers}-opcache" \
		"php${php_vers}-redis" \
		"php${php_vers}-soap" \
		"php${php_vers}-sqlite3" \
		"php${php_vers}-xdebug" \
		"php${php_vers}-xml" \
		"php${php_vers}-xmlrpc" \
		"php${php_vers}-zip"

	initsvc disablenow "php${php_vers}-fpm"
}

ubuntu_configure_php()
{
	local \
		fpm_dir="" \
		php_version=""

	php_version="$(< /opt/oct-installer/phpver)"
	fpm_dir="/etc/php/${php_version}/fpm"

	mkdir -p /var/default-conf/php-fpm
	cp -ar "${fpm_dir}"/* /var/default-conf/php-fpm/

	sed -i \
		-e "s/^listen = \/run\/.*/listen = 127.0.0.1:9000/" \
		-e "s/^user =.*/user = www-data/" \
		-e "s/^group =.*/group = www-data/" "${fpm_dir}/pool.d/www.conf"

	# reuse the session folder structure from CentOS ( originally #21069 )
	mkdir -p /var/lib/session
	chown www-data:www-data /var/lib/session
	chmod 770 /var/lib/session

	cat >> "${fpm_dir}/pool.d/www.conf" <<- EOF

		php_admin_value[post_max_size] = 2G
		php_admin_value[upload_max_filesize] = 2G
		php_admin_value[date.timezone] = UTC
		php_admin_value[error_log] = /var/log/php-fpm/www-error.log
		php_admin_value[session.save_path] = /var/lib/session
		php_admin_flag[log_errors] = on

	EOF

	mkdir -p /var/log/php-fpm
	touch /var/log/php-fpm/www-error.log
	chown -R www-data:www-data /var/log/php-fpm
}

ubuntu_optimize_php()
{
	local \
		php_version="" \
		total_mem=""

	php_version="$(< /opt/oct-installer/phpver)"
	total_mem="$(awk '/^MemTotal:/ {print $2}' /proc/meminfo)"

	if [[ -n "${total_mem}" ]] && ((total_mem < 2097152)); then
		sed -i \
			-e 's/^pm =.*/pm = dynamic/' \
			-e 's/^pm.max_children =.*/pm.max_children = 20/' \
			-e 's/^pm.start_servers =.*/pm.start_servers = 2/' \
			-e 's/^pm.min_spare_servers =.*/pm.min_spare_servers = 2/' \
			-e 's/^pm.max_spare_servers =.*/pm.max_spare_servers = 10/' \
			-e 's/^pm.max_requests =.*/pm.max_requests = 200/' "/etc/php/${php_version}/fpm/pool.d/www.conf"
	fi
}

ubuntu_install_mariadb()
{
	install_packages mariadb-server

	# backup default confs
	mkdir -p /var/default-conf/mysql
	cp /etc/mysql/mariadb.cnf /var/default-conf/mysql/mariadb.cnf

	cat >> /etc/mysql/mariadb.conf.d/60-octenium.cnf <<- EOF
		[mysqld]
		# * Cloud Provider Settings
		#
		#
		# Remove leading # and set to the amount of RAM for the most important data cache in MySQL. Start at 70% of total RAM for dedicated server, else 10%. Default is 128M.
		#innodb_buffer_pool_size=128M
		#
		# Set the number of open tables. This has a huge effect on memory usage. Default value is 2000.
		#table_open_cache=2000
	EOF

	initsvc disablenow mariadb
}

# configures memory usage settings for mysql in my.cnf
# param string mysql_innodb_buffer_pool_size
# param string mysql_table_open_cache
ubuntu_configure_mariadb()
{
	local \
		innodb_buffer_pool_size="${1:-0}" \
		table_open_cache="${2:-0}"

	[[ "${innodb_buffer_pool_size}" == "0" ]] ||
		sed -i -e "s/.*innodb_buffer_pool_size.*/innodb_buffer_pool_size=${innodb_buffer_pool_size}M/" \
			/etc/mysql/mariadb.conf.d/60-octenium.cnf

	[[ "${table_open_cache}" == "0" ]] ||
		sed -i -e "s/.*table_open_cache.*/table_open_cache=${table_open_cache}/" \
			/etc/mysql/mariadb.conf.d/60-octenium.cnf
}

ubuntu_secure_mariadb()
{
	local service_running=""
	service_running="$(
		set -euo pipefail
		initsvc isrunning mariadb
	)"

	if [[ "${service_running}" == false ]]; then
		initsvc start mariadb
	fi

	dbrootpass="$(
		set -euo pipefail
		generate_db_password
	)"

	# Make sure that NOBODY can access the server without a password
	mysql -e "SET PASSWORD FOR 'root'@localhost = PASSWORD('${dbrootpass}')"

	cat > /root/.my.cnf <<- EOF
		[client]
		user=root
		password=${dbrootpass}
	EOF

	mysql -e "FLUSH PRIVILEGES"
	# Any subsequent tries to run queries this way will get access denied because lack of usr/pwd param

	if [[ "${service_running}" == true ]]; then
		initsvc stop mariadb
	fi
}

ubuntu_install_phpmyadmin()
{
	local php_version=""
	php_version="$(< /opt/oct-installer/phpver)"

	install_packages \
		"php${php_version}-mcrypt" \
		 phpmyadmin \
		 apache2-utils

	mkdir -p /var/www/html
	ln -s /usr/share/phpmyadmin /var/www/html/mysqladmin

	if [[ ! -d /opt/oct-installer && -n "${PACKER_HTTP_ADDR:-}" ]]; then
		APP_SHORT="phpmyadmin" download_app_file phpmyadmin.conf /etc/dbconfig-common/phpmyadmin.conf
		APP_SHORT="phpmyadmin" download_app_file phpmyadmin_nginx.conf /etc/nginx/site-extras/phpymadmin.conf
	elif [[ -d /opt/oct-installer ]]; then
		mv /opt/oct-installer/phpmyadmin/phpmyadmin.conf /etc/dbconfig-common/phpmyadmin.conf
		mv /opt/oct-installer/phpmyadmin/phpmyadmin_nginx.conf /etc/nginx/site-extras/phpmyadmin.conf
	fi
}

ubuntu_configure_phpmyadmin()
{
	local \
		pma_db_user="${1:?}" \
		pma_db_pass="${2:?}" \
		pma_db_name=""

	pma_db_name="phpmy$(
		set -euo pipefail
		generate_random_characters 7 digit
	)"

	mysql --defaults-file=/root/.my.cnf -u root \
		-e "CREATE DATABASE ${pma_db_name} CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci;"
	mysql --defaults-file=/root/.my.cnf -u root \
		-e "CREATE USER '${pma_db_user}'@'localhost' IDENTIFIED BY '${pma_db_pass}';"
	mysql --defaults-file=/root/.my.cnf -u root \
		-e "GRANT ALL ON *.* TO '${pma_db_user}'@'localhost' WITH GRANT OPTION; FLUSH PRIVILEGES;"

	sed -i \
		-e "s/{USER}/${pma_db_user}/" \
		-e "s/{PASS}/${pma_db_pass}/" /etc/dbconfig-common/phpmyadmin.conf

	dpkg-reconfigure --frontend=noninteractive phpmyadmin
}

ubuntu_install_maldet()
{
	install_packages inotify-tools
	git clone https://github.com/rfxn/linux-malware-detect.git
	(
		cd linux-malware-detect
		bash ./install.sh
	)
	rm -rf ./linux-malware-detect

	sed -i \
		-e 's/quarantine_hits="0"/quarantine_hits="1"/' \
		-e 's/# default_monitor_mode/default_monitor_mode/g' \
		-e 's/default_monitor_mode="users"/# default_monitor_mode="users"/' \
		-e 's/inotify_sleep="15"/inotify_sleep="1"/' \
		-e 's/scan_ignore_root="1"/scan_ignore_root="0"/' \
		-e 's/scan_clamscan="1"/scan_clamscan="0"/' /usr/local/maldetect/conf.maldet

	printf '%s\n%s\n' "/home" "/root" >> /usr/local/maldetect/monitor_paths

	[[ -d /var/www/html ]] &&
		printf '%s\n' "/var/www/html" >> /usr/local/maldetect/monitor_paths
}

################################################################
# Common RHEL Functions
################################################################

rhel_enable_selinux()
{
	initsvc enable auditd
	if [[ "${SELINUX:-1}" == 1 ]]; then
		sed -i -e 's/^SELINUX=.*$/SELINUX=enforcing/' /etc/selinux/config
	fi
}

rhel_install_common()
{
	local package_list=(
		"cmake"
		"ethtool"
		"initscripts"
		"jq"
		"kernel-devel"
		"kernel-headers"
		"libglvnd-devel"
		"man"
		"mesa-libGL-devel"
		"nano"
		"openssh-clients"
		"parted"
		"pkg-config"
		"python3"
		"python3-pip"
		"rsync"
		"strace"
		"unzip"
		"vim"
		"watchdog"
		"wget"
		"yum-utils"
		"zlib"
	)

	case "${OCTENIUM_COMMON_OSID}" in
		almalinux* | centos* | fedora* | rocky*) package_list+=("dnf-plugin-post-transaction-actions") ;;&
		almalinux* | centos9* | rocky*) package_list+=("@Development Tools" "dhclient" "epel-release") ;;
		fedora41) ;;
		fedora*) package_list+=("@Development Tools" "dhclient" "python3-devel") ;;
		*) ;;

	esac

	write_systemd_resolved_conf

	if ! update-crypto-policies --set DEFAULT:SHA1 > /dev/null 2>&1; then
		: "Non-fatal error encountered when running 'update-crypto-policies'"
	fi

	install_packages "${package_list[@]}"
	clean_packages

	cat > /etc/dhcp/dhclient-enter-hooks <<- EOF
		#!/bin/sh
		make_resolv_conf() {
			:
		}
	EOF
	chmod +x /etc/dhcp/dhclient-enter-hooks

	chcon -h system_u:object_r:usr_t:s0 /usr/share/dict/common.*
	enforce_secure_root_passwords "/etc/pam.d/system-auth"

	initsvc disablenow postfix

	rhel_enable_selinux
}

rhel_cleanup_common()
{
	case "${OCTENIUM_COMMON_OSID}" in
		centos* | fedora*) : "Specifying to exclude from default case" ;;
		*) remove_packages "insights-client" ;;
	esac

	if ! /usr/sbin/restorecon -rv /; then
		: "Non-fatal error encountered when running 'restorecon'"
	fi

	rm -rf \
		/etc/sysconfig/network-scripts/ifcfg-enp0s2 \
		/root/{anaconda,original}-ks.cfg \
		/root/ks-{pre,post,post-update}.log

	touch /.autorelabel
}

rhel_install_docker()
{
	case "${OCTENIUM_COMMON_OSID}" in
		fedora*) os_id="fedora" ;;&
		almalinux* | centos* | rocky*) os_id="centos" ;;&
		almalinux8* | rocky8*) remove_packages buildah podman ;;&
		*) dnf config-manager --add-repo "https://download.docker.com/linux/${os_id}/docker-ce.repo" ;;
	esac

	install_packages \
		containerd.io \
		docker-buildx-plugin \
		docker-ce \
		docker-ce-cli \
		docker-compose-plugin

	useradd -g docker -m docker
}

################################################################
# Common Deb Functions
################################################################

deb_apt_set_suggests_recommends()
{
	local recommends="${1:-1}" suggests="${2:-0}"

	cat > /etc/apt/apt.conf.d/99Recommended <<- EOF
		APT::Install-Suggests "${suggests}";
		APT::Install-Recommends "${recommends}";
	EOF
}

deb_build_initramfs()
{
	local modules=(
		"megaraid_sas"
		"mpt3sas"
		"raid_class"
		"scsi_transport_sas"
		"usbhid"
		"xhci_hcd"
		"xhci_pci"
		"xhci_pci_renesas"
	)

	if [[ "${TYPE:-os}" =~ ^amd.*$ ]]; then
		modules+=("bnxt_en" "bnxt_re")
	fi

	printf '%s\n' "${modules[@]}" >> /etc/initramfs-tools/modules
	update-initramfs -u -k all
}

deb_install_common()
{
	local package_list=(
		"apache2-utils"
		"build-essential"
		"cmake"
		"ethtool"
		"gcc"
		"jq"
		"libegl1"
		"libglvnd-dev"
		"liblapack-dev"
		"libopenblas-dev"
		"libpam-pwquality"
		"lsb-release"
		"man"
		"nano"
		"parted"
		"pkg-config"
		"python3-dev"
		"python3-jsonschema"
		"python3-pip"
		"rsync"
		"smartmontools"
		"strace"
		"unzip"
		"vim"
		"watchdog"
		"wget"
		"zip"
		"zlib1g"
	)

	case "${OCTENIUM_COMMON_OSID}" in
		debian*)
			package_list+=(
				"debconf-utils"
				"dirmngr"
				"dnsutils"
				"locales-all"
				"net-tools"
				"rdnssd"
				"sudo"
				"tcpdump"
				"ufw"
			)

			if [[ "${OCTENIUM_COMMON_ARCHITECTURE}" == x86_64 ]]; then
				package_list+=("linux-headers-amd64")
			fi
			;;
		ubuntu22* | ubuntu24*)
			if [[ "${TYPE:-os}" =~ ^amd.*$ ]]; then
				package_list+=(
					"automake"
					"autoconf"
					"ibverbs-utils"
					"infiniband-diags"
					"nfs-common"
					"libibverbs-dev"
					"libtool"
					"perftest"
				)
			fi
			;;&
		ubuntu24*) package_list+=("isc-dhcp-client") ;;
		*) ;;
	esac

	install_packages "${package_list[@]}"
	clean_packages

	enforce_secure_root_passwords "/etc/pam.d/common-password"

	# #41395 Fix systemd/udev issue where serial is not in attr on attach
	cat > /etc/udev/rules.d/99-octenium-fix-virtio.rules <<- "EOF"
		KERNEL=="vd*[!0-9]", ACTION=="add", RUN+="/usr/sbin/udevadm trigger --name-match=$env{DEVNAME}"
		KERNEL=="vd*[0-9]", ACTION=="add", RUN+="/usr/sbin/udevadm trigger --name-match=$env{DEVNAME}"
	EOF
	chattr +i /etc/udev/rules.d/99-octenium-fix-virtio.rules

	case "${OCTENIUM_COMMON_OSID}" in
		debian12) write_resolv_conf ;;&
		debian11)
			write_systemd_resolved_conf
			initsvc enablenow systemd-resolved
			;;&
		debian*)
			# setup locale
			sed -i 's/^en_US ISO-8859-1/en_US.UTF-8 UTF-8/' /etc/locale.gen
			dpkg-reconfigure --frontend=noninteractive locales
			locale-gen en_US en_US.UTF-8
			;;&
		ubuntu22.04* | ubuntu24.04*)
			# PROV-728 - Ubuntu LTS releases (22.04 onward) -
			# lower default dhclient timeout from 5 minutes for systems with many ethernet interfaces
			sed -i -e 's/^timeout.*$/timeout 120;/' /etc/dhcp/dhclient.conf
			;;&
		ubuntu24*)
			rm -f /etc/systemd/system/ssh.service.requires/ssh.socket
			initsvc disable ssh.socket
			initsvc enable ssh.service
			;;&
		ubuntu*)
			sed -i \
				-e 's/GRUB_TIMEOUT_STYLE=hidden/GRUB_TIMEOUT_STYLE=menu/' \
				-e 's/GRUB_TIMEOUT=0/GRUB_TIMEOUT=3/' \
				-e '/^GRUB_DISABLE_OS_PROBER=/aGRUB_VIDEO_BACKEND=0\nGRUB_GFXMODE=text\nGRUB_GFXPAYLOAD_LINUX=text' \
				-e '/^GRUB_TIMEOUT=/aGRUB_RECORDFAIL_TIMEOUT=5/' \
				/etc/default/grub

			sed -i -e 's/#Storage=auto/Storage=auto/' /etc/systemd/journald.conf
			initsvc enablenow systemd-journald
			;;&
		*)
			sed -i -e 's/ipv6\.disable=1//' /etc/default/grub
			update-grub
			;;
	esac
}

deb_late_install_common()
{
	case "${OCTENIUM_COMMON_ARCHITECTURE}" in
		arm64)
			case "${OCTENIUM_COMMON_OSID}" in
				ubuntu22.04*) install_packages linux-nvidia-64k-hwe-22.04 ;;
				ubuntu24.04*) install_packages linux-nvidia-64k ;;
				*) ;;
			esac
			;;
		*) ;;
	esac
}

deb_install_docker()
{
	local \
		arch="${OCTENIUM_COMMON_ARCHITECTURE}" \
		docker_key_file="/usr/share/keyrings/official-docker.gpg" \
		os=""

	case "${OCTENIUM_COMMON_OSID}" in
		debian*) os="debian" ;;
		ubuntu*) os="ubuntu" ;;
		*) invalid_os_selection ;;
	esac

	curl -fsSL "https://download.docker.com/linux/${os}/gpg" | gpg --dearmor > "${docker_key_file}"

	. /etc/os-release

	if [[ "${arch}" == x86_64 ]]; then
		arch="amd64"
	fi

	printf "deb [arch=%s signed-by=%s] %s %s %s\n" \
		"${arch}" \
		"${docker_key_file}" \
		"https://download.docker.com/linux/${os}" \
		"${VERSION_CODENAME:?'unable to determine version_codename in deb_install_docker'}" \
		"stable" > /etc/apt/sources.list.d/official-docker.list

	install_packages \
		containerd.io \
		docker-buildx-plugin \
		docker-ce \
		docker-ce-cli \
		docker-compose-plugin

	useradd -g docker -m docker
}

deb_cleanup_common()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*)
			chpasswd -e <<< "ubuntu:*"
			rm -rf \
				/etc/sudoers.d/ubuntu \
				/root/up{date,grade}.txt \
				/var/lib/landscape/landscape-sysinfo.cache \
				/var/log/installer/*
			;;
		*) ;;
	esac
}

################################################################
# Common Arch Functions
################################################################

arch_build_initramfs()
{
	local modules=(
		"nvme"
		"mpt3sas"
		"raid_class"
		"scsi_transport_sas"
		"usbhid"
		"xhci_hcd"
		"xhci_pci"
		"xhci_pci_renesas"
	)

	sed -i -e "s/MODULES=()/MODULES=(${modules[*]})/g" /etc/mkinitcpio.conf
	mkinitcpio -p linux-lts
}

arch_install_common()
{
	timedatectl set-ntp true

	rm -rf /etc/NetworkManager/conf.d/*

	mkdir -p /etc/resolvconf/resolv.conf.d
	touch /etc/resolvconf/resolv.conf.d/base

	write_network_manager_dhcp_client_conf
	write_systemd_resolved_conf

	initsvc disablenow systemd-networkd

	install_packages \
		base-devel \
		bind-tools \
		cronie \
		ebtables \
		ethtool \
		expect \
		git \
		openbsd-netcat \
		gptfdisk \
		iotop \
		iptables-nft \
		jq \
		linux-lts-headers \
		linux-firmware \
		lsof \
		man-db \
		nano \
		net-tools \
		nfs-utils \
		pacutils \
		parted \
		python-pip \
		reflector \
		screen \
		sudo \
		ufw \
		unzip \
		vim \
		wget \
		zip

	clean_packages

	configure_dhclient

	if [[ -n "${APP:-}" && ! "${APP}" =~ ^plesk.*$ ]]; then
		install_packages libpwquality

		sed --follow-symlinks -i \
			-e "1apassword\trequisite\tpam_pwquality.so retry=3 minlen=8 difok=3" "/etc/pam.d/passwd"
		enforce_secure_root_passwords "/etc/pam.d/passwd"
	fi

	local aur_packages=("watchdog")

	arch_setup_installer
	if [[ "${TYPE:-"os"}" =~ ^nvidia.*$ ]]; then
		aur_packages+=("nvidia-container-toolkit")
	fi
	arch_install_pkgs "${aur_packages[@]}"
	arch_remove_installer
}

arch_cleanup_common()
{
	# Generate new mirrorlist of 50 most recently synchronized mirrors within 24 hours via HTTPS
	reflector --latest 50 --age 24 --protocol https --save /etc/pacman.d/mirrorlist

	systemctl set-default multi-user.target

	rm -rf /var/log/arch{-install,boot}.log
}

arch_refresh_pacman_keys()
{
	rm -rf /etc/pacman.d/gnupg/*.gpg
	pacman-key --init
	pacman-key --populate
	install_packages "archlinux-keyring"
}

arch_setup_installer()
{
	if ! [[ -f /etc/sudoers.d/00-octenium-installer ]]; then
		printf '%s' '%wheelnpw ALL=(ALL) NOPASSWD: ALL' > /etc/sudoers.d/00-octenium-installer
		chmod 0440 /etc/sudoers.d/00-octenium-installer
	fi

	if ! getent group wheelnpw &> /dev/null; then
		groupadd wheelnpw
	fi

	useradd --create-home --home-dir /home/installer --groups wheelnpw installer

	# Install trizen
	runuser -l installer -c '(
		git clone https://aur.archlinux.org/trizen.git
		cd trizen
		makepkg -si --noconfirm
	)'
}

arch_install_pkgs()
{
	runuser -l installer -c "trizen --sync --refresh --noconfirm $*"
}

arch_remove_installer()
{
	groupdel wheelnpw
	until ! getent group wheelnpw &> /dev/null && ! userdel --force --remove installer; do
		echo "Installer failed to remove, trying again"
		sleep 1
	done
	rm -f /etc/sudoers.d/00-octenium-installer
}

arch_install_docker()
{
	install_packages \
		"docker" \
		"docker-buildx" \
		"docker-compose"

	useradd -g docker -m docker
}

################################################################
# Common Alpine Functions
################################################################

alpine_install_common()
{
	install_packages \
		bash-completion \
		bash-doc \
		ca-certificates \
		chrony \
		doas \
		docs \
		e2fsprogs-extra \
		ethtool \
		file \
		ifupdown-ng \
		iproute2-minimal \
		libpwquality \
		man-pages \
		mandoc \
		mandoc-apropos \
		mdadm \
		parted \
		pciutils \
		python3 \
		sfdisk \
		sgdisk \
		shadow \
		sudo \
		ufw \
		usbutils \
		vim \
		wipefs

	clean_packages

	configure_dhclient

	sed -i \
		-e "1apassword\trequisite\tpam_pwquality.so retry=3 minlen=8 difok=3" "/etc/pam.d/base-password"
	enforce_secure_root_passwords "/etc/pam.d/base-password"

	# disable respawning serial console that generates errors in /var/log/messages
	sed -i \
		-e '/^ttyS0::.*vt100$/s/ttyS0/#ttyS0/' /etc/inittab
}

alpine_build_initramfs()
{
	sed -i \
		-e '/^features/s/features="\(.*\)"$/features="\1 lvm nvme raid"/' /etc/mkinitfs/mkinitfs.conf
	mkinitfs
}

################################################################
# Common OpenSUSE Functions
################################################################

opensuse_install_common()
{
	install_packages \
		bash-completion \
		ethtool \
		firewalld \
		jq \
		less \
		man \
		man-pages \
		mdadm \
		nvme-cli \
		systemd-coredump \
		vim

	clean_packages

	printf 'add_dracutmodules+=" mdraid "\n' > /etc/dracut.conf.d/mdraid.conf

	rm -f /etc/pam.d/common-password
	sed \
		-e '/^#.*/d' \
		-e '/^password.*requisite.*pam_cracklib\.so/s/pam_cracklib\.so/pam_pwquality.so/' \
		/etc/pam.d/common-password-pc > /etc/pam.d/common-password

	enforce_secure_root_passwords "/etc/pam.d/common-password"

	: > /etc/udev/rules.d/70-persistent-net.rules

	write_systemd_resolved_conf
}

opensuse_cleanup_common()
{
	rm -rf /var/adm/autoinstall
}

opensuse_install_docker()
{
	install_packages \
		docker \
		docker-buildx \
		docker-compose

	useradd -g docker -m docker
}

################################################################
# Common Functions
################################################################

download_file()
{
	curl --fail --location --silent --create-dirs --output "${2:-${1##*/}}" \
		"${DOWNLOAD_FILE_HOST_DIR:-http://${PACKER_HTTP_ADDR:-}}/${1:?}"
}

download_app_file()
{
	download_file "app/${APP_SHORT:?}/${1:?}" "${2:-}"
}

download_app_appboot()
{
	download_app_file "appboot.sh" "${1:-}"
}

download_app_provision()
{
	download_app_file "provision_app.sh" "${1:-}"
}

download_app_version()
{
	download_app_file "version.sh" "${1:-}"
}

download_os_file()
{
	download_file "${OS:?}/${1:?}" "${2:-}"
}

download_os_shared_file()
{
	download_file "${OS_SHORT:?}/${1:?}" "${2:-}"
}

download_multiple_files()
{
	for filename; do
		download_file "${filename}"
	done
}

download_multiple_app_files()
{
	download_multiple_files "${@/#/app/${APP_SHORT:?}/}"
}

download_multiple_os_files()
{
	download_multiple_files "${@/#/${OS:?}/}"
}

download_multiple_os_shared_files()
{
	download_multiple_files "${@/#/${OS_SHORT:?}/}"
}

download_universal_file()
{
	DOWNLOAD_FILE_HOST_DIR="http://10.0.2.2:5555/image_http" download_file "${1:?}" "${2:-}"
}

download_mirrored_file()
{
	DOWNLOAD_FILE_HOST_DIR="https://imagebuilder-isos.vultrlabs.com" download_file "${1:?}" "${2:-}"
}

run_file()
{
	local \
		filename="${1:?}" \
		keep_file="${2:-}"

	if [[ -n "${filename}" ]]; then
		download_file "${filename}"
		bash "./${filename##*/}"

		if [[ -z "${keep_file}" ]]; then
			rm -f "./${filename##*/}"
		fi
	fi
}

run_app_file()
{
	run_file "app/${APP_SHORT:?}/${1:?}" "${2:-}"
}

run_app_provision()
{
	download_app_appboot /root/appboot.sh
	run_app_file "provision_app.sh"
}

download_passwordlist_files()
{
	(
		mkdir -p /usr/share/dict
		cd /usr/share/dict/
		APP_SHORT="passwordlist" download_multiple_app_files common.{hwm,pwd,pwi}
	)
}

query_metadata()
{
	local \
		var_name="${1:?}" \
		var_path="${2:-}" \
		var_val=""

	if ! var_val="$(
		curl --fail --silent --header "Metadata-Token: vultr" \
			"http://169.254.169.254/${var_path:-"v1/internal/app-${var_name}"}" 2> /dev/null
	)"; then
		: "Error in query_metadata. Received '${var_val}'."
	fi

	eval "${var_name}=\${var_val}"
}

# populates a global variable with an optional app variable from the VM
# param string var_name
# param string var_path (optional)
get_var_opt()
{
	query_metadata "$@"
}

# populates a global variable with an app variable from the VM
# param string var_name
# param string var_path (optional)
get_var() {
    local name="$1"
    local file="/var/lib/cloud/seed/nocloud-net/meta-data"

    if [ ! -f "$file" ]; then
        echo "meta-data not found at $file" >&2
        exit 1
    fi

    value=$(grep -E "^${name}:" "$file" | head -n1 | cut -d':' -f2- | xargs)

    if [ -z "$value" ]; then
        echo "Variable '$name' not found in $file" >&2
        exit 1
    fi

    eval "$name=\"$value\""
}


get_ip()
{
    
ip="$(curl -s https://files.oct-backend.com/octeniumvz/ip.php)"
    
}

get_ipv6()
{
	get_var_opt ipv6 "meta-data/meta-data/ipv6-addr"
}

get_public_interface()
{
	local ip="" iface=""

	get_ip
	iface="$(ip -oneline -family inet -brief address show | awk "/${ip}/ {print \$1}")"
	echo "${iface}"
}

get_ipv6_assigned_to_public_interface()
{
	local ipv6="" full_address="" public_iface=""
	get_ipv6
	if [[ -n "${ipv6}" ]]; then
		public_iface="$(
			set -euo pipefail
			get_public_interface
		)"

		if [[ -n "${public_iface}" ]]; then
			full_address="$(
				ip -oneline -family inet6 -brief address show scope global dev "${public_iface}" |
					awk "/${ipv6%:*}/ {print \$3}"
			)"
		fi
	fi

	echo "${full_address%/*}"
}

generate_certs()
{
	local \
		cert_name="${1:-"server"}" \
		cert_subj="${2:-"/CN=localhost"}" \
		cert_dir="${3:-"/etc/nginx/ssl"}" \
		cert_expiry="${4:-365}"

	mkdir -p "${cert_dir}"

	openssl req \
		-subj "${cert_subj}" \
		-new \
		-newkey rsa:2048 \
		-sha256 \
		-days "${cert_expiry}" \
		-nodes \
		-x509 \
		-keyout "${cert_dir}/${cert_name}.key" \
		-out "${cert_dir}/${cert_name}.crt"

	chmod -R 600 "${cert_dir}"
}

generate_self_signed_cert()
{
	generate_certs "$@"
}

create_root_gpg_dir()
{
	mkdir -p /root/.gnupg
	chmod 700 /root/.gnupg
}

get_architecture()
{
	case "$(uname -m)" in
		aarch64 | arm64) echo "arm64" ;;
		*) echo "x86_64" ;;
	esac
}

get_os()
{
	if [[ -n "${OS:-}" ]]; then
		local os="${OS//_/}"
		echo "${os//-/}"
	elif [[ -e /etc/os-release ]]; then
		. /etc/os-release
		case "${ID:-}" in
			alpine | arch) echo "${ID}linux" ;;
			*) echo "${ID}${VERSION_ID:-}" ;;
		esac
	else
		echo ""
	fi
}

get_flavor()
{
	case "${OCTENIUM_COMMON_OSID}" in
		almalinux* | centos* | cloudlinux* | fedora* | rocky*) echo "rhel" ;;
		alpinelinux) echo "alpinelinux" ;;
		archlinux) echo "archlinux" ;;
		debian* | ubuntu*) echo "deb" ;;
		opensuse*) echo "opensuse" ;;
		*) invalid_os_selection ;;
	esac
}

get_rootfs_device()
{
	local rootfs_device=""
	if ! rootfs_device="$(findmnt --noheadings --output=source --raw --notruncate /)" &&
		[[ -z "${rootfs_device}" ]]; then
		echo "Failed to determine rootfs device!"
		exit 255
	fi

	echo "${rootfs_device}"
}

generate_random_characters()
{
	local \
		num_chars="${1:-"16"}" \
		char_type="${2:-"alnum"}"

	case "${char_type}" in
		alnum | alpha | digit | graph | lower | print | punct | upper | xdigit)
			(
				set +euxo pipefail
				/usr/bin/tr --delete --complement "[:${char_type}:]" < /dev/urandom |
					/usr/bin/head --bytes="${num_chars}"
			)
			;;
		*) printf "Invalid character type set (%s)!\n" "${char_type}" ;;
	esac

}

generate_db_password()
{
	generate_random_characters 15 alnum
}

cockpit_gitlab_fix()
{
	sed -i -e 's/9090/9091/g' /usr/lib/systemd/system/cockpit.socket
	initsvc daemonreload
}

prepare_appboot()
{
	if [[ -f /root/appboot.sh ]]; then
		mkdir -p /var/lib/cloud/scripts/per-once
		mv /root/appboot.sh /var/lib/cloud/scripts/per-once/
		chmod +x /var/lib/cloud/scripts/per-once/appboot.sh
	fi
}

prepare_app_image()
{
	local app="${APP_SHORT%-*}"

	if [[ "${TYPE:-}" == app || "${APP_SHORT:-}" != "${OS_SHORT:-}" ]]; then
		APP_SHORT="${app}" download_app_version /opt/octenium//version.sh
		APP_SHORT="${app}" run_app_provision
	fi
}

toggle_unattended_upgrades()
{
	local action="${1:?}" apt_conf_file="/etc/apt/apt.conf.d/9999-disable-unattended-upgrades"

	if [[ "${OCTENIUM_COMMON_FLAVOR}" == deb ]]; then
		until ! lsof -t /var/cache/apt/archives/lock /var/lib/apt/lists/lock /var/lib/dpkg/lock > /dev/null 2>&1; do
			sleep 2
		done

		case "${action}" in
			disable) printf '%s\n' 'APT::Periodic::Unattended-Upgrade "0";' > "${apt_conf_file}" ;;
			enable) [[ -f "${apt_conf_file}" ]] && rm -f "${apt_conf_file}" ;;
			*)
				echo "Invalid 'action' specified for 'toggle_unattended_upgrades()'"
				exit 255
				;;
		esac
	fi
}

write_resolv_conf()
{
	# Use internal and quad9.net DNS servers
	cat > /etc/resolv.conf <<- EOF
		nameserver 108.61.10.10
		nameserver 9.9.9.9
		nameserver 2001:19f0:300:1704::6
		nameserver 2620:fe::fe
	EOF
}

write_systemd_resolved_conf()
{
	cat > /etc/systemd/resolved.conf <<- EOF
		[Resolve]
		DNS=108.61.10.10 2001:19f0:300:1704::6
		FallbackDNS=9.9.9.9 2620:fe::fe
		ReadEtcHosts=yes
	EOF
}

write_version_sh()
{
	cat > /opt/octenium//version.sh <<- "EOF"
		#!/bin/bash
		set -euo pipefail

		. /etc/os-release
		os_version="${VERSION:-${VERSION_ID:-$(uname -r)}}"
		echo "'OS: ${os_version}'"
	EOF
}

write_network_manager_dhcp_client_conf()
{
	cat > /etc/NetworkManager/conf.d/00-dhcp-client.conf <<- EOF
		[main]
		dhcp=dhclient
	EOF
}

append_oct_app_bashrc()
{
	if [[ -f "${HOME}/.bashrc" ]]; then
		printf '\n%s\n' '[[ -f /opt/octenium/oct_app.sh ]] && . /opt/octenium/oct_app.sh' >> "${HOME}/.bashrc"
	fi
}

disable_numa_balancing()
{
	cat > /usr/lib/sysctl.d/90-octenium-gpu.conf <<- EOF
		kernel.numa_balancing=0
	EOF
}

write_infinite_memlock_ulimit()
{
	mkdir -p /etc/systemd/system.conf.d
	cat > /etc/systemd/system.conf.d/99-octenium.conf <<- EOF
		[Manager]
		DefaultLimitMEMLOCK=infinity
	EOF
}

force_load_net_modules()
{
	local mod_names=(
		"bnxt_en"
		"ixgbe"
		"mlx5_core"
	)
	printf '%s\n' "${mod_names[@]}" >> /etc/modules-load.d/net.conf
}

modules_load_nvidia()
{
	cat > /etc/modules-load.d/nvidia.conf <<- EOF
		nvidia
		nvidia_peermem
		nvidia_vgpu_vfio
	EOF
}

blacklist_nouveau()
{
	cat > /etc/modprobe.d/blacklist-nouveau.conf <<- EOF
		blacklist nouveau
		options nouveau modeset=0
	EOF
}

nvidia_module_options()
{
	cat > /etc/modprobe.d/nvidia-options.conf <<- EOF
		options nvidia NVreg_EnableMSI=0 NVreg_OpenRmEnableUnsupportedGpus=1
	EOF
}

blacklist_ast()
{
	if [[ "${OCTENIUM_COMMON_ARCHITECTURE}" == arm64 || "${TYPE:-os}" =~ ^amdbm$ ]]; then
		return
	fi

	cat > /etc/modprobe.d/blacklist-ast.conf <<- EOF
		blacklist ast
		blacklist astdrm
		blacklist astdrmfb
	EOF
}

blacklist_ipmi()
{
	cat > /etc/modprobe.d/blacklist_ipmi.conf <<- EOF
		blacklist ipmi_ssif
		install ipmi_ssif /bin/true

		blacklist ipmi_si
		install ipmi_si /bin/true

		blacklist ipmi_devintf
		install ipmi_devintf /bin/true

		blacklist ipmi_msghandler
		install ipmi_msghandler /bin/true
	EOF
}

blacklist_mei()
{
	cat > /etc/modprobe.d/blacklist-mei.conf <<- EOF
		blacklist mei_me
	EOF
}

blacklist_rndis()
{
	cat > /etc/modprobe.d/blacklist-rndis_host.conf <<- EOF
		blacklist rndis_host
	EOF
}

dracut_build_initramfs()
{
	local nvme_drivers=(
		"ixgbe"
		"megaraid_sas"
		"mpt3sas"
		"nvme"
		"raid1"
		"raid_class"
		"scsi_transport_sas"
	)

	printf 'force_drivers+=" %s "\n' "${nvme_drivers[*]}" > /etc/dracut.conf.d/nvme.conf
	chattr +i /etc/dracut.conf.d/nvme.conf

	dracut_output="$(dracut -f 2>&1)"
	if grep -q -s FAILED <<< "${dracut_output}" > /dev/null 2>&1; then
		printf 'Error in dracut: %s\n' "${dracut_output}"
		exit 255
	fi
}

enforce_secure_root_passwords()
{
	local \
		pam_conf="${1:?PAM configuration file path required}" \
		pw_pam_lib="pam_pwquality\.so" \
		pw_pam_rule="try_first_pass local_users_only authtok_type= minlen=8 retry=3 difok=3 enforce_for_root"

	sed -i -e "s/# dictpath =/dictpath =\/usr\/share\/dict\/common/" /etc/security/pwquality.conf

	sed -i \
		-e "/^password.*${pw_pam_lib}/ s/${pw_pam_lib}.*$/${pw_pam_lib} ${pw_pam_rule}/" \
		-e "/^password.*pam_unix\.so.*use_authtok/!{/^auth.*pam_unix\.so/!s/pam_unix\.so\(.*\)/pam_unix\.so\1 use_authtok/}" "${pam_conf}"
}

disable_ipv6_privacy_extensions()
{
	cat > /usr/lib/sysctl.d/90-octenium-ipv6-privacy.conf <<- EOF
		net.ipv6.conf.all.use_tempaddr = 0
		net.ipv6.conf.default.use_tempaddr = 0
	EOF
}

ensure_hostname_set_for_cloud_init()
{
	mkdir -p /var/lib/cloud/data
	cat > /var/lib/cloud/data/set-hostname <<- EOF
		{
		    "fqdn": "guest.domain.com",
		    "hostname": "guest"
		}
	EOF
}

manipulate_sshd_config()
{
	sed -i \
		-e 's/^PermitRootLogin/#PermitRootLogin/g' \
		-e '/^ListenAddress ::/d' \
		-e '/^ListenAddress 0\.0\.0\.0/d' \
		-e '1aPermitRootLogin yes\nListenAddress 127.0.0.1\n' /etc/ssh/sshd_config
}

enable_discard()
{
	local rootfs_device=""
	rootfs_device="$(
		set -euo pipefail
		get_rootfs_device
	)"

	tune2fs -o discard "${rootfs_device}"
}

break_raid()
{
	local md_regex='^md[[:digit:]]+$' disk_to_eject="" rootfs_device=""
	rootfs_device="$(
		set -euo pipefail
		get_rootfs_device
	)"

	if [[ "${rootfs_device##*/}" =~ ${md_regex} ]]; then
		disk_to_eject="$(realpath -e -P -q "/sys/devices/virtual/block/${rootfs_device##*/}/md/dev-vdb"*/block)"
		mdadm --manage "${rootfs_device}" --fail "/dev/${disk_to_eject##*/}"
		sleep 2
		sync
		sleep 2
		sync
		mdadm --manage "${rootfs_device}" --remove "/dev/${disk_to_eject##*/}"
	fi
}

configure_firewall_firewalld()
{
	initsvc enablenow firewalld

	case "${OCTENIUM_COMMON_OSID}" in
		fedora*) firewall-cmd --set-default-zone=FedoraServer ;;
		*) firewall-cmd --set-default-zone=public ;;
	esac
}

configure_firewall_ufw()
{
	chmod 600 /etc/ufw/*.rules
	(
		set +eo pipefail
		ufw --force reset
		ufw default deny
		ufw allow 22/tcp
		ufw --force enable
	)

	initsvc enable ufw
}

install_pcie_disable_acs_service()
{
	cat > /var/lib/oct//pcie_disable_acs.sh <<- "EOF"
		#!/bin/bash

		while read -r addr _; do
		    if ! setpci -f -r -s "${addr}" ECAP_ACS+0x6.w > /dev/null 2>&1; then
		        : "${addr} does not support ACS. skipping."
		        continue
		    fi

		    if ! setpci -f -r -s "${addr}" ECAP_ACS+0x6.w=0000 > /dev/null 2>&1; then
		        echo "Error disabling ACS on ${addr}"
		        continue
		    fi

		    if ! new="$(setpci -f -r -s "${addr}" ECAP_ACS+0x6.w 2> /dev/null)" || [[ "${new}" != 0000 ]]; then
		        echo "ACS register did not return disabled state on ${addr}"
		        continue
		    fi
		done < <(lspci -d '*:*:*' || :)
	EOF
	chmod 440 /var/lib/oct//pcie_disable_acs.sh

	mkdir -p /var/lib/oct//services
	cat > /var/lib/oct//services/pcie-disable-acs.service <<- EOF
		[Unit]
		Description=Disable PCIe ACS

		[Service]
		Type=oneshot
		ExecStart=/bin/bash /var/lib/oct//pcie_disable_acs.sh

		[Install]
		WantedBy=multi-user.target
	EOF

	chmod 644 /var/lib/oct//services/pcie-disable-acs.service
	initsvc enable /var/lib/oct//services/pcie-disable-acs.service
}

install_pkey_service()
{
	cat > /var/lib/oct//pkey_manager.sh <<- "EOF"
		#!/bin/bash

		function configure_interface()
		{
		    ip link set dev "${1}" up
		    echo "${2}" > "/sys/class/net/${1}/create_child"
		    ip link set dev "${1}.${2:2}" up
		    echo "Configured ${1} with ${2}"
		}

		if [[ ! -f /sys/class/infiniband/mlx5_0/ports/1/pkeys/0 ]]; then
		    echo "No Infiniband detected"
		    exit 0
		fi

		pkey="$(< /sys/class/infiniband/mlx5_0/ports/1/pkeys/0)"
		echo "Waiting for PKEY to be set"
		retries=25
		while [[ "${pkey}" == "0x7fff" && "${retries}" -gt 0 ]]; do
		    sleep 5
		    pkey="$(< /sys/class/infiniband/mlx5_0/ports/1/pkeys/0)"
		    retries=$((retries - 1))
		done

		if [[ "${pkey}" == "0x7fff" ]]; then
		    echo "PKEY not set"
		    exit 1
		fi

		myPKEY="$(printf '0x%04X\n' "$((pkey ^ 0x8000))")"
		echo "Pkey found: ${myPKEY}"

		for interface in $(ip -br l | awk '/^ib/ {print $1}' || true); do
		    # This can fail if ran multiple times
		    configure_interface "${interface}" "${myPKEY}" || true
		done
		echo "Done configuring interfaces for Infiniband"
	EOF
	chmod 440 /var/lib/oct//pkey_manager.sh

	mkdir -p /var/lib/oct//services
	cat > /var/lib/oct//services/octenium-pkey-manager.service <<- EOF
		[Unit]
		Wants=network-online.target
		After=network-online.target
		After=pcie-disable-acs.service
		After=nvidia-fabricmanager.service
		Description=Configure PKeys for Infiniband and set up interfaces

		[Service]
		Type=oneshot
		ExecStart=/bin/bash /var/lib/oct//pkey_manager.sh

		[Install]
		WantedBy=multi-user.target
	EOF

	chmod 644 /var/lib/oct//services/octenium-pkey-manager.service
	initsvc enable /var/lib/oct//services/octenium-pkey-manager.service
}

install_broadcom_netxtreme_e_drivers()
{
	# https://techdocs.broadcom.com/us/en/storage-and-ethernet-connectivity/ethernet-nic-controllers/bcm957xxx/adapters/software-installation/installing-the-linux-driver/manual-software-installation-and-configuration.html
	find /usr/lib64 -name libbnxt_re-rdmav\*.so -exec mv -v '{}' '{}.bak' \;
	find /usr/lib -name libbnxt_re-rdmav\*.so -exec mv -v '{}' '{}.bak' \;

	download_mirrored_file broadcom-netxtreme-e.zip
	unzip -qq broadcom-netxtreme-e.zip
	install_packages \
		./bcm*/drivers_linux/bnxt_en/dkms/bnxt-en-dkms*_all.deb \
		./bcm*/drivers_linux/bnxt_re/dkms/bnxt-re-dkms*_all.deb \
		./bcm*/utils/niccli/linux_x86_64/sliff/dkms/sliff-dkms*_all.deb \
		./bcm*/utils/niccli/linux_x86_64/niccli*_x86_64.deb

	(
		cd ./bcm*/drivers_linux/bnxt_rocelib
		tar -xzvf libbnxt_re-*.tar.gz
		cd libbnxt_re-*/
		bash ./autogen.sh
		./configure --sysconfdir=/etc
		make
		make install all
		if ! grep '^/usr/local/lib$' /etc/ld.so.conf /etc/ld.so.conf.d/*.conf; then
			printf '/usr/local/lib\n' > /etc/ld.so.conf.d/libbnxt.conf
			ldconfig
		fi

		cp bnxt_re.driver /etc/libibverbs.d/
	)

	rm -rf broadcom-netxtrem-e.zip bcm*

	write_cloud_init_find_candidate_nics_helper

	cat > /usr/lib/udev/rules.d/60-rdma-persistent-network-names.rules <<- EOUDEVRULES
		ACTION=="add", SUBSYSTEM=="infiniband", PROGRAM="rdma_rename %k NAME_KERNEL"
	EOUDEVRULES
}

blank_root_password()
{
	chpasswd -e <<< "root:*"
}

cleanup_cloud_init()
{
	rm -rf \
		/etc/cloud \
		/etc/systemd/system/cloud-init.target.wants/* \
		/lib/systemd/system/cloud* \
		/run/cloud-init \
		/usr/bin/cloud* \
		/usr/lib/cloud* \
		/usr/local/bin/cloud* \
		/usr/src/cloud* \
		/var/log/cloud*

	# We need to enforce the existence of cloud-init log
	# files, otherwise cloud-init log reader will fail to start.
	touch /var/log/cloud-init.log
}

cleanup_imageless()
{
	rm -rf /opt/oct-installer 2>/dev/null
	rm -rf /opt/octenium 2>/dev/null
	touch /var/lib/oct/.application-installed
}

imgboot_exit()
{
	local rc="${1:-"0"}"
	if [[ "${rc}" = 0 ]]; then
		rm -rf /opt/imageboot
	fi

	exit "${rc}"
}

################################################################
# Dynamic functions
################################################################
invalid_os_selection()
{
	echo "Invalid OS Selection!"
	exit 255
}

invalid_package_operation()
{
	echo "Invalid package operation specified!"
	exit 255
}

invalid_init_system_operation()
{
	echo "Invalid init system operation specified!"
	exit 255
}

invalid_service_operation()
{
	echo "Invalid service operation specified!"
	exit 255
}

package_operation()
{
	local pkg_op="${1:?'No package operation specified'}"
	shift

	case "${OCTENIUM_COMMON_OSID}" in
		almalinux* | centos* | fedora* | rocky*)
			case "${pkg_op}" in
				autoremove) dnf --assumeyes autoremove ;;
				clean)
					dnf --assumeyes clean all
					rm -rf /var/cache/dnf
					;;
				install) dnf --assumeyes install "${@}" ;;
				remove) dnf --assumeyes erase "${@}" ;;
				upgrade) dnf --assumeyes upgrade "${@}" ;;
				*) invalid_package_operation ;;
			esac
			;;
		alpinelinux)
			case "${pkg_op}" in
				autoremove | clean) : "Not implemented or automatically handled by package manager" ;;
				install) apk --update-cache add "${@}" ;;
				remove) apk del --purge --rdepends "${@}" ;;
				upgrade) apk --update-cache upgrade --available "${@}" ;;
				*) invalid_package_operation ;;
			esac
			;;
		archlinux)
			case "${pkg_op}" in
				autoremove)
					local package_list=() pacman_output=""
					if pacman_output="$(
						set -euo pipefail
						pacman --query --deps --unrequired --unrequired --quiet
					)"; then
						mapfile -t package_list <<< "${pacman_output}"
						if ((${#package_list[@]} > 0)); then
							pacman --noconfirm --remove --recursive "${package_list[@]}"
						fi
					fi
					;;
				clean)
					pacman --noconfirm --sync --clean
					rm -f /var/cache/pacman/pkg/*.zst{,.sig}
					;;
				install) pacman --noconfirm --sync --refresh --overwrite='*' "${@}" ;;
				remove) pacman --noconfirm --remove "${@}" ;;
				upgrade) pacman --noconfirm --sync --refresh --sysupgrade "${@}" ;;
				*) invalid_package_operation ;;
			esac
			;;
		debian* | ubuntu*)
			export DEBIAN_FRONTEND='noninteractive' DEBIAN_PRIORITY='critical' NEEDRESTART_MODE='a'
			case "${pkg_op}" in
				autoremove | clean | install | remove | upgrade)
					until ! lsof -t \
						/var/cache/apt/archives/lock \
						/var/lib/apt/lists/lock \
						/var/lib/dpkg/lock{,-frontend} > /dev/null 2>&1; do
						: waiting 3 seconds for apt lock held by another process
						sleep 3
					done
					;;&
				install | upgrade) apt-get update ;;&
				autoremove) apt-get --assume-yes autoremove ;;
				clean)
					apt-get --assume-yes clean
					rm -rf /var/cache/apt
					;;
				install) apt-get --assume-yes install "${@}" ;;
				remove) apt-get --assume-yes purge "${@}" ;;
				upgrade) apt-get --assume-yes upgrade "${@}" ;;
				*) invalid_package_operation ;;
			esac
			;;
		opensuse*)
			case "${pkg_op}" in
				install | upgrade) zypper --non-interactive refresh ;;&
				autoremove)
					local package_list=() awk_list="" zypper_output=""
					if zypper_output="$(
						set -euo pipefail
						zypper --non-interactive --terse --quiet packages --unneeded
					)"; then
						awk_list="$(awk -F'|' '/^i \|/ {print $3}' <<< "${zypper_output}")"
						while read -r pkg; do
							package_list+=("${pkg}")
						done <<< "${awk_list}"
						if ((${#package_list[@]} > 0)); then
							zypper --non-interactive remove --no-confirm "${package_list[@]}"
						fi
					fi
					;;
				clean) zypper --non-interactive clean ;;
				install) zypper --non-interactive install --no-confirm --no-recommends "${@}" ;;
				remove) zypper --non-interactive remove --no-confirm --clean-deps "${@}" ;;
				upgrade) zypper --non-interactive update --no-confirm "${@}" ;;
				*) invalid_package_operation ;;
			esac
			;;
		*) invalid_os_selection ;;
	esac
}

autoremove_packages()
{
	package_operation autoremove
}

clean_packages()
{
	package_operation clean
}

install_packages()
{
	package_operation install "${@}"
}

remove_packages()
{
	package_operation remove "${@}"
}

upgrade_packages()
{
	package_operation upgrade
}

pyinstall_packages()
{
	local tmp_dir="/opt/octenium//tmp" pip_args=("install")
	case "${OCTENIUM_COMMON_OSID}" in
		archlinux | debian12 | ubuntu23*) pip_args+=("--break-system-packages") ;;
		*) ;;
	esac

	mkdir -p "${tmp_dir}"
	TMPDIR=${tmp_dir} python3 -m pip "${pip_args[@]}" "${@}"
	rm -rf "${tmp_dir}" /root/.cache/pip
}

init_system_operation()
{
	local \
		init_sys="${1:?'No init system specified'}" \
		svc_op="${2:?'No service operation specified'}"
	shift
	shift

	case "${init_sys}" in
		systemd)
			case "${svc_op}" in
				daemonreload) systemctl daemon-reload ;;
				disable)
					for svc; do
						if systemctl is-enabled "${svc}" > /dev/null 2>&1; then
							systemctl disable "${svc}"
						fi
					done
					;;
				disablenow)
					for svc; do
						if systemctl is-enabled "${svc}" > /dev/null 2>&1; then
							systemctl disable --now "${svc}"
						fi
					done
					;;
				enable)
					for svc; do
						if ! systemctl is-enabled "${svc}" > /dev/null 2>&1; then
							systemctl enable "${svc}"
						fi
					done
					;;
				enablenow)
					for svc; do
						if ! systemctl is-enabled "${svc}" > /dev/null 2>&1; then
							systemctl enable --now "${svc}"
						fi
					done
					;;
				isrunning)
					if (($# > 1)); then
						echo "Unable to process more than one service for 'isrunning'"
						exit 255
					fi

					local svc="${1:?'No service specified'}"
					if systemctl is-active "${svc}" > /dev/null 2>&1; then
						echo "true"
					else
						echo "false"
					fi
					;;
				mask) systemctl mask "${@}" ;;
				reload) systemctl reload "${@}" ;;
				restart) systemctl restart "${@}" ;;
				start)
					for svc; do
						if ! systemctl is-active "${svc}" > /dev/null 2>&1; then
							systemctl start "${svc}"
						fi
					done
					;;
				stop)
					for svc; do
						if systemctl is-active "${svc}" > /dev/null 2>&1; then
							systemctl stop "${svc}"
						fi
					done
					;;
				unmask) systemctl unmask "${@}" ;;
				*) invalid_service_operation ;;
			esac
			;;
		openrc)
			case "${svc_op}" in
				disable)
					for svc; do
						if rc-service --exists "${svc}"; then
							rc-update del "${svc}" default
						fi
					done
					;;
				disablenow)
					for svc; do
						if rc-service --exists "${svc}"; then
							rc-service --ifexists "${svc}" stop
							rc-update del "${svc}" default
						fi
					done
					;;
				enable)
					for svc; do
						if rc-service --exists "${svc}"; then
							rc-update add "${svc}" default
						fi
					done
					;;
				enablenow)
					for svc; do
						if rc-service --exists "${svc}"; then
							rc-update add "${svc}" default
							rc-service --ifexists "${svc}" start
						fi
					done
					;;
				reload)
					for svc; do
						rc-service --ifexists "${svc}" reload
					done
					;;
				restart)
					for svc; do
						rc-service --ifexists "${svc}" restart
					done
					;;
				start)
					for svc; do
						rc-service --ifexists "${svc}" start
					done
					;;
				stop)
					for svc; do
						rc-service --ifexists "${svc}" stop
					done
					;;
				*) invalid_service_operation ;;
			esac
			;;
		*) invalid_init_system_operation ;;
	esac
}

initsvc()
{
	case "${OCTENIUM_COMMON_OSID}" in
		almalinux*) ;&
		archlinux) ;&
		centos*) ;&
		debian*) ;&
		fedora*) ;&
		opensuse*) ;&
		rocky*) ;&
		ubuntu*) init_system_operation "systemd" "${@}" ;;
		alpinelinux) init_system_operation "openrc" "${@}" ;;
		*) invalid_os_selection ;;
	esac
}

configure_boot_parameters()
{
	local params=("consoleblank=0" "mpt3sas.max_queue_depth=1000" "vultr" "ds=octenium")

	# kdump memory requirements stolen from RHEL8 Product Documentation - Managing, monitoring, and updating the kernel
	case "${OCTENIUM_COMMON_ARCHITECTURE}" in
		arm64)
			params+=("console=ttyAMA0,115200n8")
			case "${OCTENIUM_COMMON_FLAVOR}" in
				opensuse | rhel) params+=("crashkernel=512M-2G:240M,2G-:480M") ;;
				*) ;;
			esac
			;;
		x86_64)
			params+=("console=ttyS0,115200n8")
			case "${OCTENIUM_COMMON_FLAVOR}" in
				opensuse | rhel) params+=("crashkernel=512M-1G:128M,1G-4G:192M,4G-64G:256M,64G-:512M") ;;
				*)
					case "${TYPE:-}" in
						amdbm) params+=("pci=realloc=off") ;;&
						*bm) params+=("iommu=pt") ;;
						*) ;;
					esac
					;;
			esac
			;;
		*) ;;
	esac

	add_parameters_grub_cmdline_linux "${params[@]}" "console=tty0"

	chown root:root /etc/modprobe.d/*.conf
	chmod 644 /etc/modprobe.d/*.conf

	build_initramfs
}

add_parameters_grub_cmdline_linux()
{
	local grub_param_line_sentinel="GRUB_CMDLINE_LINUX"

	case "${OCTENIUM_COMMON_OSID}" in
		almalinux* | centos* | fedora* | rocky*) grubby --update-kernel=ALL --args="${*}" ;;
		alpinelinux | archlinux | ubuntu*) grub_param_line_sentinel="GRUB_CMDLINE_LINUX_DEFAULT" ;;&
		alpinelinux | archlinux | debian* | opensuse* | ubuntu*)
			for param; do
				sed -i \
					-e "s/^${grub_param_line_sentinel}=\"\(.*\)\"/${grub_param_line_sentinel}=\"\1 ${param}\"/" \
					/etc/default/grub
			done
			;;&
		alpinelinux | archlinux) grub-mkconfig -o /boot/grub/grub.cfg ;;
		debian* | ubuntu*) update-grub ;;
		opensuse*) grub2-mkconfig -o /boot/grub2/grub.cfg ;;
		*) invalid_os_selection ;;
	esac
}

configure_ntp()
{
	local ntp_conf="" ntp_svc="chronyd" ntp_servers=(
		"1.time.constant.com"
		"2.time.constant.com"
		"3.time.constant.com"
	)

	case "${OCTENIUM_COMMON_OSID}" in
		almalinux* | centos* | fedora* | opensuse* | rocky*) ntp_conf="/etc/chrony.conf" ;;&
		alpinelinux) ntp_conf="/etc/chrony/chrony.conf" ;;&
		archlinux | ubuntu*)
			sed --follow-symlinks -i \
				-e "s/#NTP=.*/NTP=${ntp_servers[*]}/" /etc/systemd/timesyncd.conf
			ntp_svc="systemd-timesyncd"
			;;&
		debian11)
			ntp_conf="/etc/ntp.conf"
			ntp_svc="ntp"
			;;&
		debian12)
			ntp_conf="/etc/ntpsec/ntp.conf"
			ntp_svc="ntpsec"
			;;&
		almalinux* | alpinelinux | centos* | debian* | fedora* | rocky*)
			sed -i \
				-e '/^server/d' \
				-e 's/^pool/#pool/g' "${ntp_conf}"

			printf 'server %s iburst\n' "${ntp_servers[@]}" >> "${ntp_conf}"
			;;&
		almalinux* | alpinelinux | archlinux | centos* | debian* | fedora* | opensuse* | rocky* | ubuntu*)
			initsvc enablenow "${ntp_svc}"
			;;
		*) invalid_os_selection ;;
	esac
}

configure_dhclient()
{
	local dhclient_conf=
	case "${OCTENIUM_COMMON_OSID}" in
		alpinelinux) dhclient_conf="/etc/dhcp/dhclient.conf" ;;
		archlinux) dhclient_conf="/etc/dhclient.conf" ;;
		*) invalid_os_selection ;;
	esac

	APP_SHORT="dhclient-cfg" download_app_file dhclient.conf "${dhclient_conf}"
	APP_SHORT="dhclient-cfg" download_app_file rfc3442-classless-routes.sh /etc/dhclient-exit-hooks
}

configure_firewall()
{
	case "${OCTENIUM_COMMON_FLAVOR}" in
		opensuse | rhel) configure_firewall_firewalld ;;
		*) configure_firewall_ufw ;;
	esac
}

configure_watchdog()
{
	case "${OCTENIUM_COMMON_OSID}" in
		alpinelinux) ;; # only available in 'edge' testing branch repository
		opensuse*) ;;   # not packaged
		debian* | ubuntu*)
			sed -i -e 's/^watchdog_module=.*/watchdog_module="i6300esb"/' /etc/default/watchdog
			;;&
		*)
			printf '%s\n' "i6300esb" > /etc/modules-load.d/watchdog.conf
			sed -i -e '/\/dev\/watchdog/s/#watchdog-device/watchdog-device/' /etc/watchdog.conf
			initsvc enable watchdog
			;;
	esac
}

write_cloud_init_find_candidate_nics_helper()
{
	# This is hardcoded in cloud-init sources - if present, will allow overriding default interface detection for DHCP
	local script_name="/opt/octenium//find_candidate_nics.sh"
	case "${TYPE:-os}" in
		amdbm) APP_SHORT=cloud-init-dhcp-helpers download_app_file amdbm.sh "${script_name}" ;;
		*) ;;
	esac

	if [[ -s "${script_name}" ]]; then
		chmod 0540 "${script_name}"
	fi
}

install_upstream_cloud_init_via_python_setup()
{
	local cloudinit_exe=""
	if cloudinit_exe="$(command -v cloud-init 2>&1)" && [[ -x "${cloudinit_exe}" ]]; then
		remove_packages "cloud-init"
	fi

	git clone https://github.com/canonical/cloud-init.git /usr/src/cloud-init
	(
		cd /usr/src/cloud-init
		python3 -m pip install setuptools
		python3 -m pip install -r requirements.txt

		python3 setup.py build
		python3 setup.py install --init-system systemd

		if [[ ! -f /usr/bin/cloud-init && -f /usr/local/bin/cloud-init ]]; then
			ln -s /usr/local/bin/cloud-init /usr/bin/cloud-init
		fi

		if [[ ! -f /usr/bin/cloud-init ]]; then
			: "Figure out where cloud-init is getting installed"
			exit 255
		fi

		if [[ -f /etc/udev/rules.d/70-persistent-net.rules ]]; then
			chattr -i /etc/udev/rules.d/70-persistent-net.rules
		fi

		initsvc unmask cloud-init-local cloud-init cloud-config cloud-final
		initsvc enable cloud-init-local cloud-init cloud-config cloud-final
	)
}

install_upstream_cloud_init()
{
	mkdir /usr/src/cloud-init

	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*)
			. /etc/os-release
			local branch="${VERSION_CODENAME:-}" deps_out="" deps=()

			chown ubuntu:ubuntu /usr/src/cloud-init
			runuser --user ubuntu -- bash -c "
				git clone https://github.com/canonical/cloud-init.git -b ubuntu/${branch} /usr/src/cloud-init
			"

			deps_out="$(
				/usr/src/cloud-init/tools/read-dependencies \
					--requirements-file /usr/src/cloud-init/requirements.txt \
					--distro ubuntu \
					--system-pkg-names
			)"

			mapfile -t deps <<< "${deps_out}"
			install_packages ubuntu-dev-tools "${deps[@]}"
			runuser --user ubuntu -- bash -c "
				cd /usr/src/cloud-init && DEB_BUILD_OPTIONS=nocheck ./packages/bddeb -d --release ${branch}
			"

			install_packages /usr/src/cloud-init/cloud-init_all.deb

			remove_packages ubuntu-dev-tools
			autoremove_packages
			;;

		*) install_upstream_cloud_init_via_python_setup ;;
	esac
}

install_cloud_init()
{
	cloud_init_packages=("cloud-init")
	case "${OCTENIUM_COMMON_OSID}" in
		almalinux*) ;&
		alpinelinux*) ;&
		centos*) ;&
		fedora*) ;&
		rocky*) cloud_init_packages+=("cloud-utils-growpart") ;;
		archlinux*) cloud_init_packages+=("cloud-utils") ;;
		debian*) cloud_init_packages+=("cloud-guest-utils") ;;
		*) ;;
	esac

	install_packages "${cloud_init_packages[@]}"
	cloud-init --version
}

configure_cloud_init()
{
	if ! cloud-init clean --logs --seed; then
		: "Failed running 'cloud-init clean'"
		exit 255
	fi

	APP_SHORT="cloudcfg" download_app_file "cloud.cfg.${OS_SHORT:?}.yml" /etc/cloud/cloud.cfg
	APP_SHORT="cloudcfg/cloud.cfg.d" download_app_file "95_ds-octenium.cfg" /etc/cloud/cloud.cfg.d/95_ds-octenium.cfg

	case "${OCTENIUM_COMMON_OSID}" in
		alpinelinux*)
			setup-cloud-init # script provided by alpine's cloud-init package
			initsvc enable cloud-init-hotplugd
			;;
		*)
			# Disable NetworkManager-wait-online.service as it breaks booting with cloud-init
			initsvc disable NetworkManager-wait-online
			initsvc enable cloud-init-local cloud-config cloud-final

			# newer versions of cloud-init have a different unit file name - instead of tracking OSes that are using
			# the newer version, try both
			# shellcheck disable=SC2310
			initsvc enable cloud-init-main || initsvc enable cloud-init
			;;
	esac

	# We need to enforce the existence of cloud-init log
	# files, otherwise cloud-init log reader will fail to start.
	touch /var/log/cloud-init.log

	install_cloud_init_log_reader_service
	ensure_hostname_set_for_cloud_init
}

build_initramfs()
{
	case "${OCTENIUM_COMMON_FLAVOR}" in
		alpinelinux) alpine_build_initramfs ;;
		archlinux) arch_build_initramfs ;;
		deb) deb_build_initramfs ;;
		opensuse | rhel) dracut_build_initramfs ;;
		*) invalid_os_selection ;;
	esac
}

install_gpu_packages_early()
{
	case "${TYPE:-"os"}" in
		amdbm* | nvidiabm*) disable_ipv6_privacy_extensions ;;&
		amd* | nvidia*)
			install_pcie_disable_acs_service
			write_infinite_memlock_ulimit
			disable_numa_balancing
			;;&
		amdbm)
			case "${OCTENIUM_COMMON_OSID}" in
				ubuntu*) install_broadcom_netxtreme_e_drivers ;;
				*) ;;
			esac
			;;&

		amd*)
			case "${OCTENIUM_COMMON_OSID}" in
				ubuntu*)
					install_amdgpu_install
					amdgpu-install --usecase=rocm --assume-yes
					rocm_post_install
					;;&

				*) install_docker ;;
			esac
			;;

		nvidiabm)
			local gpu_package_list=()

			case "${OCTENIUM_COMMON_OSID}" in
				centos* | rocky*) gpu_package_list+=("cuda-toolkit" "kmod-nvidia-open-dkms") ;;&
				archlinux) gpu_package_list+=("cuda" "nvidia-lts" "xorg") ;;&
				ubuntu*)
					install_ofed_repo
					install_packages "mlnx-ofed-all"
					gpu_package_list+=(
						"nsight-compute-"
						"nsight-systems-"
						"nvidia-settings-"
						"cuda-toolkit"
						"libnccl-dev"
						"libnccl2"
						"nvidia-compute-utils-${NVIDIA_DRIVER_VERSION}"
						"nvidia-driver-${NVIDIA_DRIVER_VERSION}-open"
						"nvidia-fabricmanager-${NVIDIA_DRIVER_VERSION}"
						"nvidia-utils-${NVIDIA_DRIVER_VERSION}"
						"nvlsm"
						"rdma-core"
						"xserver-xorg"
					)
					;;&

				*)
					install_nvidia_repo
					modules_load_nvidia
					nvidia_module_options

					install_packages "${gpu_package_list[@]}"
					clean_packages
					;;

			esac
			;;&

		nvidia)
			# Xorg needs to be installed first otherwise xserver modules won't be installed
			case "${OCTENIUM_COMMON_OSID}" in
				archlinux) install_packages xorg ;;&
				ubuntu*) install_packages xserver-xorg ;;&
				archlinux | ubuntu*) clean_packages ;;&
				*)
					download_mirrored_file "vultr_nvidia_driver.zip" /tmp/vultr_nvidia_driver.zip
					unzip -o /tmp/vultr_nvidia_driver.zip -d /
					rm -f /tmp/vultr_nvidia_driver.zip
					;;

			esac
			;;&

		nvidia*)
			# Innoculate against automation
			touch /var/lib/oct//states/.vultr-gpu{,-script}

			install_nvidia_persistenced_service
			install_nvidia_container_toolkit
			install_ngc_cli
			;;

		*) ;;
	esac
}

install_gpu_packages_late()
{
	case "${TYPE:-"os"}" in
		nvidia)
			# Installing NVIDIA driver as late as possible to not require multiple rebuilds due to lack of GPU + hooks
			bash /opt/nvidia/install.sh

			for svc in gridd topologyd; do
				if [[ ! -s "/usr/lib/systemd/system/nvidia-${svc}.service" ]]; then
					cp "/usr/lib/nvidia/systemd/nvidia-${svc}.service" /usr/lib/systemd/system/
				fi
			done

			initsvc enable nvidia-topologyd
			;;

		nvidiabm)
			case "${OCTENIUM_COMMON_OSID}" in
				ubuntu*)
					install_pkey_service

					mkdir -p /var/lib/cloud/scripts/per-instance
					cat > /var/lib/cloud/scripts/per-instance/vultr-gpu.sh <<- "EOF"
						#!/bin/bash
						{
						    if command -v nvidia-smi && nvlink="$(nvidia-smi nvlink -s)" && [[ -n "${nvlink}" ]]; then
						        systemctl enable --now nvidia-fabricmanager.service
						    fi
						} > /dev/null 2>&1
					EOF

					chmod +x /var/lib/cloud/scripts/per-instance/vultr-gpu.sh
					;;

				*) ;;
			esac
			;;

		*) ;;
	esac
}

install_nvidia_persistenced_service()
{
	if [[ ! -f /usr/lib/systemd/system/nvidia-persistenced.service ]]; then
		groupadd nvidia-persistenced
		useradd \
			--shell /sbin/nologin \
			--home-dir /var/run/nvpd \
			--no-create-home \
			--system \
			--comment 'NVIDIA Persistence Daemon' \
			--gid nvidia-persistenced \
			nvidia-persistenced

		cat > /usr/lib/systemd/system/nvidia-persistenced.service <<- EOF
			[Unit]
			Description=NVIDIA Persistence Daemon

			[Service]
			Type=forking
			ExecStartPre=/bin/mkdir /var/run/nvidia-persistenced
			ExecStartPre=/bin/chown nvidia-persistenced:nvidia-persistenced /var/run/nvidia-persistenced
			ExecStart=/usr/bin/nvidia-persistenced --user nvidia-persistenced
			ExecStopPost=/bin/rm -rf /var/run/nvidia-persistenced

			[Install]
			WantedBy=multi-user.target
		EOF
	fi

	initsvc disable nvidia-persistenced.service
}

install_nvidia_container_toolkit()
{
	install_docker

	# Fix nvidia being nvidia and breaking its own container system https://github.com/NVIDIA/nvidia-docker/issues/1730
	local run_keys=""
	run_keys="RUN+=\"/usr/bin/nvidia-ctk system create-device-nodes --load-kernel-modules --control-devices\""
	run_keys="${run_keys}, RUN+=\"/usr/bin/nvidia-ctk system create-dev-char-symlinks --create-all\""

	cat > /etc/udev/rules.d/99-vultr-fix-nvidia.rules <<- EOF
		# This will create /dev/char symlinks to all device nodes
		ACTION=="add", DEVPATH=="/bus/pci/drivers/nvidia*", ${run_keys}
	EOF
	chattr +i /etc/udev/rules.d/99-vultr-fix-nvidia.rules

	case "${OCTENIUM_COMMON_FLAVOR}" in
		deb)
			curl -fsSL https://nvidia.github.io/libnvidia-container/gpgkey |
				gpg --dearmor > /usr/share/keyrings/nvidia-container-toolkit.gpg

			DOWNLOAD_FILE_HOST_DIR=https://nvidia.github.io/libnvidia-container/stable/deb \
				download_file nvidia-container-toolkit.list /etc/apt/sources.list.d/nvidia-container-toolkit.list

			sed -i \
				-e 's#deb https://#deb [signed-by=/usr/share/keyrings/nvidia-container-toolkit.gpg] https://#g' \
				/etc/apt/sources.list.d/nvidia-container-toolkit.list
			;;&
		rhel)
			if [[ ! "${TYPE:-"os"}" =~ ^nvidiabm$ ]]; then
				DOWNLOAD_FILE_HOST_DIR=https://nvidia.github.io/libnvidia-container/stable/rpm \
					download_file nvidia-container-toolkit.repo /etc/yum.repos.d/nvidia-container-toolkit.repo
			fi
			;;&
		deb | rhel) install_packages nvidia-container-runtime nvidia-container-toolkit nvidia-docker2 ;;&
		archlinux | deb | rhel) nvidia-ctk runtime configure --runtime=docker ;;
		*) invalid_os_selection ;;
	esac

	local cfg=""
	cfg="$(< /etc/docker/daemon.json)"
	jq -reM '{"exec-opts":["native.cgroupdriver=cgroupfs"],"runtimes":.runtimes}' <<< "${cfg}" > /etc/docker/daemon.json
}

rocm_post_install()
{
	if [[ "${OCTENIUM_COMMON_ARCHITECTURE}" != x86_64 ]]; then
		return
	fi

	cat > /etc/ld.so.conf.d/rocm.conf <<- EOF
		/opt/rocm/lib
		/opt/rocm/lib64
	EOF

	ldconfig

	sed -i -e '/^PATH="/s/PATH="\(.*\)"/PATH="\1:\/opt\/rocm\/bin"/' /etc/environment
}

install_amdgpu_install()
{
	if [[ "${OCTENIUM_COMMON_ARCHITECTURE}" != x86_64 ]]; then
		return
	fi

	local codename=""

	case "${OCTENIUM_COMMON_FLAVOR}" in
		deb)
			case "${OCTENIUM_COMMON_OSID}" in
				ubuntu22*) codename="jammy" ;;
				ubuntu24*) codename="noble" ;;
				*) invalid_os_selection ;;
			esac
			;;
		*) ;;
	esac

	download_mirrored_file "${codename}-amdgpu-install.deb"
	install_packages "./${codename}-amdgpu-install.deb"
	rm -f "./${codename}-amdgpu-install.deb"
}

install_nvidia_repo()
{
	local arch="${OCTENIUM_COMMON_ARCHITECTURE}" cuda_keyring_version="1.1-1" nvos=""

	if [[ "${arch}" == arm64 ]]; then
		arch="sbsa"
	fi

	case "${OCTENIUM_COMMON_OSID}" in
		archlinux*) ;;
		centos* | rocky*)
			nvos="rhel9"
			repo_file=""
			yum-config-manager --add-repo \
				"https://developer.download.nvidia.com/compute/cuda/repos/${nvos}/${arch}/cuda-${nvos}.repo"
			;;
		ubuntu20*) nvos="ubuntu2004" ;;&
		ubuntu22*) nvos="ubuntu2204" ;;&
		ubuntu24*) nvos="ubuntu2404" ;;&
		ubuntu*)
			DOWNLOAD_FILE_HOST_DIR="https://developer.download.nvidia.com/compute/cuda/repos/${nvos}/${arch}" \
				download_file "cuda-keyring_${cuda_keyring_version}_all.deb"
			install_packages "./cuda-keyring_${cuda_keyring_version}_all.deb"
			rm -f "cuda-keyring_${cuda_keyring_version}_all.deb"
			;;
		*) invalid_os_selection ;;
	esac
}

install_ofed_repo()
{
	local \
		key_file="/usr/share/keyrings/mlnx_ofed.gpg" \
		list_file="/etc/apt/sources.list.d/mellanox_mlnx_ofed.list" \
		mlnxos=""

	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu20*) mlnxos="ubuntu20.04" ;;
		ubuntu22*) mlnxos="ubuntu22.04" ;;
		ubuntu24*) mlnxos="ubuntu24.04" ;;
		*) invalid_os_selection ;;
	esac

	curl -fsSL https://www.mellanox.com/downloads/ofed/RPM-GPG-KEY-Mellanox | gpg --dearmor > "${key_file}"
	DOWNLOAD_FILE_HOST_DIR="https://linux.mellanox.com/public/repo/mlnx_ofed/latest/${mlnxos}" \
		download_file mellanox_mlnx_ofed.list "${list_file}"

	sed -i -e "s#^deb http#deb [signed-by=${key_file}] http#" "${list_file}"
}

install_docker()
{
	if ! command -v docker > /dev/null 2>&1; then
		case "${OCTENIUM_COMMON_FLAVOR}" in
			archlinux) arch_install_docker ;;
			deb) deb_install_docker ;;
			opensuse) opensuse_install_docker ;;
			rhel) rhel_install_docker ;;
			*) invalid_os_selection ;;
		esac
	fi

	initsvc enable docker
}

install_ngc_cli()
{
	download_mirrored_file "ngc-cli-${OCTENIUM_COMMON_ARCHITECTURE}.zip"
	unzip -qq "ngc-cli-${OCTENIUM_COMMON_ARCHITECTURE}.zip"
	mv ngc-cli /usr/local/bin/
	rm -rf "./ngc-cli-${OCTENIUM_COMMON_ARCHITECTURE}.zip" ./ngc-cli.md5

	cat > /etc/profile.d/90_vultr_ngc_cli.sh <<- 'EOF'
		export PATH="${PATH}:/usr/local/bin/ngc-cli"
	EOF
}

install_kubectl()
{
	local arch="${OCTENIUM_COMMON_ARCHITECTURE}" release repo_file version
	if [[ "${arch}" != "arm64" ]]; then
		arch="amd64"
	fi

	release="$(curl -fsL https://dl.k8s.io/release/stable.txt)"

	if [[ -n "${release}" && "${release}" =~ ^v[[:digit:]]+\.[[:digit:]]+\.[[:digit:]]+$ ]]; then
		case "${OCTENIUM_COMMON_FLAVOR}" in
			deb)
				repo_file="/etc/apt/sources.list.d/kubernetes.list"
				version="${release%.*}"
				curl -fsSL "https://pkgs.k8s.io/core:/stable:/${version}/deb/Release.key" |
					gpg --dearmor -o /etc/apt/keyrings/kubernetes-apt-keyring.gpg
				chmod 644 /etc/apt/keyrings/kubernetes-apt-keyring.gpg

				printf 'deb [arch=%s signed-by=%s] %s %s\n' \
					"${arch}" \
					"/etc/apt/keyrings/kubernetes-apt-keyring.gpg" \
					"https://pkgs.k8s.io/core:/stable:/${version}/deb/" \
					"/" > "${repo_file}"
				chmod 644 "${repo_file}"
				;;

			opensuse) repo_file="/etc/zypp/repos.d/kubernetes.repo" ;;&
			rhel) repo_file="/etc/yum.repos.d/kubernetes.repo" ;;&
			opensuse | rhel)
				cat > "${repo_file}" <<- EOF
					[kubernetes]
					name=Kubernetes
					baseurl=https://pkgs.k8s.io/core:/stable:/${version}/rpm/
					enabled=1
					gpgcheck=1
					gpgkey=https://pkgs.k8s.io/core:/stable:/${version}/rpm/repodata/repomd.xml.key
				EOF
				;;

			*) invalid_os_selection ;;
		esac

		install_packages kubectl
	else
		echo "Unable to determine stable k8s release version!"
		exit 255
	fi
}

install_k3s_stack()
{
	(
		set -euo pipefail
		curl -sfL https://get.k3s.io | sh -
	)

	install_kubectl

	(
		set -euo pipefail
		curl https://raw.githubusercontent.com/helm/helm/main/scripts/get-helm-3 | bash
	)
}

install_cockpit()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*)
			ubuntu_install_cockpit
			(($# == 0)) || ubuntu_configure_cockpit "$@"
			;;
		*) invalid_os_selection ;;
	esac
}

configure_cockpit()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*) ubuntu_configure_cockpit "$@" ;;
		*) invalid_os_selection ;;
	esac
}

install_nginx()
{
	install_packages "nginx"
	mkdir -p \
		/etc/nginx/htpasswd \
		/etc/nginx/modules-available \
		/etc/nginx/modules-enabled \
		/etc/nginx/orig \
		/etc/nginx/site-extras \
		/etc/nginx/sites-available \
		/etc/nginx/sites-enabled \
		/var/default-conf/nginx \
		/var/www/html

	for orig_conf in /etc/nginx/nginx.conf /etc/nginx/conf.d/default.conf; do
		if [[ -e "${orig_conf}" ]]; then
			mv "${orig_conf}" /var/default-conf/nginx/
		fi
	done

	cat > /etc/nginx/nginx.conf <<- EOF
		user nginx;
		worker_processes auto;
		pid /run/nginx.pid;
		include /etc/nginx/modules-enabled/*.conf;

		events {
		    worker_connections 2000;
		}

		http {
		    sendfile on;
		    tcp_nopush on;
		    types_hash_max_size 2048;
		    server_tokens off;

		    include /etc/nginx/mime.types;
		    default_type application/octet-stream;

		    ssl_protocols TLSv1 TLSv1.1 TLSv1.2 TLSv1.3;
		    ssl_prefer_server_ciphers on;

		    access_log /var/log/nginx/access.log;
		    error_log /var/log/nginx/error.log;

		    gzip on;
		    gzip_vary on;
		    gzip_proxied any;
		    gzip_comp_level 6;
		    gzip_buffers 16 8k;
		    gzip_http_version 1.1;
		    gzip_types text/plain text/css application/json application/javascript text/xml application/xml application/xml+rss text/javascript;

		    include /etc/nginx/sites-enabled/*.conf;
		}
	EOF

	cat > /opt/octenium/fix-vhost.sh <<- EOF
		#!/bin/bash

		set -euo pipefail
		shopt -s inherit_errexit > /dev/null 2>&1 || :

		mkdir -p /etc/nginx/sites-available.bak

		if mv -f /etc/nginx/sites-available/*.conf /etc/nginx/sites-available.bak/; then
		    if cp /etc/nginx/orig/*.conf /etc/nginx/sites-available/; then
		        systemctl restart nginx.service
		        echo "All vhosts have been restored to their default state!"
		    fi
		fi
	EOF

	cat > /opt/octenium/remove-htaccess.sh <<- EOF
		#!/bin/bash

		set -euo pipefail
		shopt -s inherit_errexit > /dev/null 2>&1 || :

		if sed -i -e 's/auth_basic.*#SAFE TO REMOVE//g' /etc/nginx/{conf.d,sites-enabled}/*.conf; then
		    systemctl restart nginx.service
		    echo "All basic authentication has been removed!"
		fi
	EOF

	chmod +x /opt/octenium/fix-vhost.sh /opt/octenium//remove-htaccess.sh
	initsvc disablenow nginx

	local server_user="nginx"
	case "${OCTENIUM_COMMON_OSID}" in
		archlinux) server_user="http" ;;&
		debian* | ubuntu*) server_user="www-data" ;;&
		archlinux | debian* | ubuntu*) sed -i -e "s/^user nginx;/user ${server_user};/" /etc/nginx/nginx.conf ;;
		*) ;;
	esac
}

install_certbot()
{
	case "${OCTENIUM_COMMON_OSID}" in
		archlinux) install_packages "certbot-nginx" ;;
		almalinux* | centos* | debian* | fedora* | rocky* | ubuntu*) install_packages "python3-certbot-nginx" ;;
		*) invalid_os_selection ;;
	esac
}

# We only want this on BM and not virtual machines
install_fbdev()
{
	local proc_manufacturer=""
	proc_manufacturer="$(dmidecode -s processor-manufacturer)"
	if [[ "${proc_manufacturer}" != QEMU ]] && ! command -v nvidia-smi > /dev/null 2>&1; then
		mkdir -p /etc/X11/xorg.conf.d/
		cat > /etc/X11/xorg.conf.d/10-fbdev.conf <<- EOF
			Section "Device"
				Identifier "fb0"
				Driver "fbdev"
			EndSection
		EOF
	fi
}

install_php()
{
	local php_version="${1:?"Specify desired PHP version: 8.1|8.2|8.3"}" optimize="${2:-}" invalid_version="false"

	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*)
			case "${php_version}" in
				8.1) ubuntu_install_php 8.1 ;;&
				8.2) ubuntu_install_php 8.2 ;;&
				8.3) ubuntu_install_php 8.3 ;;&
				8.1 | 8.2 | 8.3)
					ubuntu_configure_php

					if [[ -n "${optimize}" ]]; then
						ubuntu_optimize_php
					fi

					;;
				*) invalid_version="true" ;;
			esac
			;;
		*) invalid_os_selection ;;
	esac

	if [[ "${invalid_version}" == "true" ]]; then
		printf "Invalid PHP version (%s) specified for this OS (%s)\n" "${php_version}" "${OCTENIUM_COMMON_OSID}"
		exit 255
	fi
}

optimize_php()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*) ubuntu_optimize_php ;;
		*) invalid_os_selection ;;
	esac
}

install_mysql()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*)
			ubuntu_install_mariadb
			ubuntu_configure_mariadb "$@"
			ubuntu_secure_mariadb
			;;
		*) invalid_os_selection ;;
	esac
}

# param string mysql_innodb_buffer_pool_size
# param string mysql_table_open_cache
configure_mysql()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*) ubuntu_configure_mariadb "$@" ;;
		*) invalid_os_selection ;;
	esac
}

secure_mysql()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*) ubuntu_secure_mariadb ;;
		*) invalid_os_selection ;;
	esac
}

install_phpmyadmin()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*)
			ubuntu_install_phpmyadmin
			(($# == 0)) || ubuntu_configure_phpmyadmin "$@"
			;;
		*) invalid_os_selection ;;
	esac
}

configure_phpmyadmin()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*) ubuntu_configure_phpmyadmin "$@" ;;
		*) invalid_os_selection ;;
	esac
}

install_maldet()
{
	case "${OCTENIUM_COMMON_OSID}" in
		ubuntu*) ubuntu_install_maldet ;;
		*) invalid_os_selection ;;
	esac
}

distro_install_common()
{
	case "${OCTENIUM_COMMON_FLAVOR}" in
		alpinelinux) alpine_install_common ;;
		archlinux) arch_install_common ;;
		deb) deb_install_common ;;
		opensuse) opensuse_install_common ;;
		rhel) rhel_install_common ;;
		*) : "No common for OS flavor?" ;;
	esac
}

distro_late_install_common()
{
	case "${OCTENIUM_COMMON_FLAVOR}" in
		deb) deb_late_install_common ;;
		*) : "No late install common for OS flavor" ;;
	esac
}

distro_cleanup_common()
{
	if [[ "${UEFI:-}" == yes ]]; then
		case "${OCTENIUM_COMMON_OSID}" in
			archlinux) grub-install --bootloader-id=archlinux --efi-directory=/boot/efi --removable ;;&
			debian*) grub-install --removable ;;&
			opensuse*) grub2-install --removable ;;&
			*) umount /boot/efi ;;
		esac
	fi

	for trunc_file in /etc/machine-id /var/log/auth.log /var/log/secure; do
		if [[ -f "${trunc_file}" ]]; then
			: > "${trunc_file}"
		fi
	done

	case "${OCTENIUM_COMMON_FLAVOR}" in
		archlinux) arch_cleanup_common ;;&
		deb) deb_cleanup_common ;;&
		opensuse) opensuse_cleanup_common ;;&
		rhel) rhel_cleanup_common ;;&
		*)
			rm -rf \
				/etc/NetworkManager/system-connections/*.nmconnection \
				/etc/ufw/*.rules.* \
				/opt/octenium//tmp \
				/tmp/vultr \
				/var/lib/cloud/seed \
				/var/lib/dbus/machine-id \
				/var/lib/dhclient/*.lease \
				/var/lib/dhcp/*.leases \
				/var/lib/NetworkManager/*.lease \
				/var/lib/systemd/random-seed \
				/var/lib/oct//states/provisioning \
				/var/log/dmesg*
			;;
	esac
}

install_common()
{
	uname -a
	date > /var/log/image_build_date
	mkdir -p /etc/cloud /opt/octenium// /var/lib/oct/{config,services,states}
	download_file "app/vultrbuilder.sh" /var/lib/oct/vultrbuilder.sh

	touch /var/lib/oct//states/provisioning

	break_raid

	force_load_net_modules

	blacklist_ast
	blacklist_ipmi
	blacklist_mei
	blacklist_rndis
	blacklist_nouveau

	manipulate_sshd_config

	download_passwordlist_files

	append_oct_app_bashrc

	write_version_sh
	write_resolv_conf

	adjust_repos

	clean_packages

	distro_install_common
	configure_firewall

	install_cloud_init
	install_gpu_packages_early

	configure_ntp
	# configure_watchdog

	configure_boot_parameters
}

cleanup_common()
{
	upgrade_packages

	configure_cloud_init

	# Any per-once or similar scripts need to be installed/configured AFTER 'cloud-init clean'.
	# Otherwise, they just get erased
	prepare_app_image

	distro_late_install_common

	autoremove_packages
	clean_packages

	run_file "app/version.sh"

	install_gpu_packages_late

	distro_cleanup_common

	enable_discard
	blank_root_password

	sync
	sync
	sync

	df -hlPT

	case "${OCTENIUM_COMMON_OSID}" in
		alpinelinux) poweroff ;;
		*) shutdown -h +1 ;;
	esac

	echo "build complete"
}

packer_provision()
{
	install_common
	cleanup_common
}

install_cloud_init_log_reader_service()
{
	case "${OCTENIUM_COMMON_OSID}" in
		alpinelinux)
			cat > /opt/octenium//cloud-init-log-reader.sh <<- EOF
				#!/bin/sh
				LESSSECURE=1 /usr/bin/less /var/log/cloud-init.log
			EOF
			chmod 500 /opt/octenium//cloud-init-log-reader.sh

			sed -i \
				-e 's/^tty4::.*/tty4::respawn:\/sbin\/getty -L -n -i -l \/opt\/vultr\/cloud-init-log-reader.sh 38400 tty4/' \
				/etc/inittab
			;;
		*)
			# Uses systemd
			cat > /var/lib/oct//services/cloud-init-log-reader.service <<- EOF
				[Unit]
				Description=Cloud-Init TTY4 Log Reader
				After=getty@tty4.service

				[Service]
				Type=simple
				Environment=LESSSECURE=1
				ExecStart=/usr/bin/less /var/log/cloud-init.log
				Restart=always
				StandardInput=tty
				StandardOutput=tty
				TTYPath=/dev/tty4
				TTYReset=yes
				TTYVHangup=yes

				[Install]
				WantedBy=multi-user.target
			EOF
			chmod 644 /var/lib/oct//services/cloud-init-log-reader.service

			initsvc enable /var/lib/oct//services/cloud-init-log-reader.service
			;;
	esac
}

adjust_repos()
{
	case "${OCTENIUM_COMMON_OSID}-${OCTENIUM_COMMON_ARCHITECTURE}" in
		alpinelinux*)
			sed -E -i \
				-e '/.*community$/s/^#http/http/' \
				-e '/^http.*\/edge\/[[:alpha:]]+$/s/http/#http/' /etc/apk/repositories
			;;
		archlinux*)
			cat >> /etc/pacman.conf <<- EOF
				[multilib]
				Include = /etc/pacman.d/mirrorlist

				[multilib-testing]
				Include = /etc/pacman.d/mirrorlist
			EOF

			sed -i \
				-e 's/\#Color/Color/g' \
				-e 's/\#ParallelDownloads = 5/ParallelDownloads = 50/g' \
				-e "s|\[options\]|[options]\nDisableDownloadTimeout|" /etc/pacman.conf
			;;
		debian*)
			debian_write_sources_list
			write_app_repo
			;;&
		fedora*x86_64)
			for repo_file in fedora{,-updates{,-testing}}.repo; do
				APP_SHORT="mirrors/fedora" download_app_file "${repo_file}" "/etc/yum.repos.d/${repo_file}"
			done
			;;&
		rocky*)
			# For now, Rescale wants their RCE images to run a specific vaulted version - ideally this is temporary
			if [[ "${APP_SHORT:-}" == rce ]]; then
				(
					. /etc/os-release
					sed -i \
						-e 's/^mirrorlist/#mirrorlist/' \
						-e 's/^#baseurl/baseurl/' \
						-e "/^baseurl/s/\$contentdir\/\$releasever/vault\/rocky\/${VERSION_ID:?}/" \
						/etc/yum.repos.d/Rocky-*.repo
				)
			fi
			;;&
		ubuntu*) deb_apt_set_suggests_recommends ;;&
		ubuntu*x86_64)
			if [[ -f /etc/os-release ]]; then
				. /etc/os-release
			fi

			local codename="${VERSION_CODENAME:?'Codename not found in /etc/os-release'}"
			rm -f /etc/apt/sources.list
			APP_SHORT="mirrors/ubuntu" download_app_file "sources.list" /etc/apt/sources.list
			sed -i -e "s/{{RELEASE}}/${codename}/g" /etc/apt/sources.list
			;;&
		almalinux* | centos* | fedora* | rocky*)
			if [[ "${TYPE:-"os"}" =~ ^nvidia.*$ ]]; then
				sed -i -e '/\[main\]/ainstall_weak_deps=false' /etc/dnf/dnf.conf
			fi
			;;
		debian* | ubuntu*)
			if [[ "${TYPE:-"os"}" =~ ^nvidia.*$ ]]; then
				deb_apt_set_suggests_recommends 0
			fi
			;;
		*) : "No repo configuration needed for this OS." ;;
	esac
}

# Writes the "vultr-apprepo" GPG key to the distribution specific file, dearmoring on Debian-based distros
write_app_repo()
{
	local arch="${OCTENIUM_COMMON_ARCHITECTURE}" gpg_file="" os_name="" temp_file=""

	case "${OCTENIUM_COMMON_FLAVOR}" in
		deb | rhel)
			temp_file="$(mktemp)"

			cat > "${temp_file}" <<- EOF
				-----BEGIN PGP PUBLIC KEY BLOCK-----
				Version: GnuPG v2.0.22 (GNU/Linux)

				mQINBGIo0nEBEAC4tnw40/LZOuMGzC3h4LnT9oS5c++J6Pf7/IihFxVRCbV9L5O5
				lhVYIJS5YTo8yfWAuvTUzXge85GuAVzUjckKJjiNAhxTQ0CRnvrLhLe0id1+WKWC
				wy8x6PC6IY+56qvBXbrsxqaTwMPEN4hQDJxoqGIFvFz2BLfKRNOVbpk7tKQDy7x8
				6oMUFpppLAJ8Q8xH0q7v03R3FdaMHTWHIbL/+Mu9DYEySTLFfgGVAhelgL4kScEB
				XMf0LLc8va29Y9n20B+ncwlr100RA45s7fHM82e1vDlwb6YQftByPf30JB3RlkJh
				srkRztXbbrJy5EQ1m0monTbe6JjppdxhooU5rXDY18Cx3qJ+iuBs04ocC34UFjvw
				amFH5bjYM/rwMRTBu7TT68v9fnEQCldY4FpD1HiUXm3KnL7rBSAt+SIvrsFeoRlH
				ga3KS5bWy9PPuA/eQpGvd6q8LRSHcGx76vMAM5f5vfA8M5lNw2VBxROJ6v2zIGaj
				HSEm3srJ/D4XYvJtgWceBweZa65qxfBt5sJlfRdNq1c7awCJeph5Aen1vW+XIzhS
				0LDYkMgqzkbFzw83izQeuPypNYoHDyX/tj0TZVKG6C6NJWLb8PDWVzUYdCDRP+wp
				i+JT8F+IzgrSTv0PBHFSjNn7JinT+6b8KzxVW8AOzXBVeHWaTkaleIyjDQARAQAB
				tB5WdWx0ciBBcHBzIDxzdXBwb3J0QHZ1bHRyLmNvbT6JAj8EEwECACkFAmIo0nEC
				GwMFCRLMAwAHCwkIBwMCAQYVCAIJCgsEFgIDAQIeAQIXgAAKCRDUsX3rXs2yUz7s
				D/9/OMma0MWhW5F5296fB/EMv/D4Bno0kC+eDc3PxQ0c49eWU733abSeFhC04fF0
				1KtsIVzXyaL+S+k2Lv/paZKxWK/1u+9FVnjI+B/iiKomoBqVWUdSqOPDJUYub1Gm
				XKhf+jDjRz1FeVfCY+4+uV9x4ItkNNvUV9z0jyn4QWnAx3vNijb8gbr8w1vCnEtG
				7AoZZUtuctT76ryHaNz9J00ucpf+kU5YUAcwlAYLO0QWLFetYzX1btsT8UBXnGm1
				wUD4cVoKSdY0GrYvme81ZiiZ7Ump3HUtPJ9dsO52EsOuEdFo+gMSteSjkco13M9k
				vbwhruYdxZv/TTwnUGHGVFvxAB8uS//jvjOBjJbn6/RJIusZywtzYXZYfOW5wXIn
				3ZUJbPpsFeaB49ukBSa3ZnbUmf/M5w/jH4vfqjn8slspT2ByjNof6Is0lnRpK6tM
				dN4AduRo59Pj5Ma7vye6w2wffSBw8WlHsPWsl5D41aSkzO9x8fhYeu3zv0fQ0JsO
				PPKEkT3/C5sqqvR/s+Tf0LhTAvTAHg52CRFFzmc6MOXLz2GjJteHAaJ6eCsraPa3
				c7djG5O530S0C0UoQA6gollSZY8ngGp/P9sD6S4ohPc7JEvxD3H1w4OaL1ECusEt
				tKJGT395ivsqxAAbIX0IIJEoLMBYH1cItWy3iZ/UzHNyh7kCDQRiKNJxARAA17uk
				HCdiE24672HzU8vwAeHJlDua7adEovfdxLkJeV8wI7seZiGVNfktZZ71VPGy+nnM
				gJVy0PqJyI5BljUqP/0Me2Ij4brTtrtHtm1KLGsp56nvrtUR61fBnMNqINwpJfPh
				0dQirDMU8hr8y/51I7J+hBYucoWEipm/MACBFkWFeDnr63lrzItavK0+v5XWpXkp
				IqPXpJqW9JRn/JR3bOzVN1X7TtuEEg6suoPmCCvZu5C6tGLTcdbiq4ga4VvxUb/d
				q7ukgfYRgDo1OI0/BK2OiE97ux5C3nm15NvrIMFJ0GBwhdvJPWpqL1NkYi3iiX0+
				qgpdVxQ1ySULdZcnEOKCkIBv3bBQPQiGPsRXCKdV+UjxUv7sjdKmKDLdzW+KgEvw
				43MqeXfCFPk0x6tlFkJtEAXH/DLuesf6BzNCKKDeZdw0YaXPeEQFHprm0zbjvR2w
				3vlFVOeXZlWzGFQJJoX72+HepO0pKEZfunxQN6wkg85SR/IgOaTz1bdB/QsvbBLD
				Xc4SI6i99Kk5DYBugRaH+onqOS0QEguWm3Y8EK79pxEHdOW1DcgNZShMZXhniAvx
				hQ6Xc1Kpb7wVPxjixAm8GfVHz04ytik9+GrfdSnmvnRZ9GP+8i3xQmoY+A1pHvHy
				K5t9di6V0+SA/cBmRDbFOwgwJVGn82RPaSQU6lsAEQEAAYkCJQQYAQIADwUCYijS
				cQIbDAUJEswDAAAKCRDUsX3rXs2yUwzUD/9o36kXUEJ6lFysIbT4L4K+N41OQo+A
				emYRCE/Yh0dtpS3runW2JDPeRDjoXFgl5ldaLw62NgqInhW/guEZVc6sMko6Hgft
				pa1tlqRWqnEr4ho1zpdb8dci5n051tAYWGTGsFP8nbnhN/JVisrJRdXdsjz+hZ8y
				rhE/dcm5+7DXLJM7Km/NwnodFcOhRC57noO5UssMeG7otSoRcffpfsPDgdsY+VGC
				9Br+ODQ/oRFHXCHoa+B8jOjCDc8lK4+q8iBplZy55CC12hGKx9Ra8yI9vUWVqG/j
				4H50E9zUtGoAfEMYf4Xy4srvwpZSbqqW94EDydfZfVXevxdGbreb844AJk6xyjY9
				YaVcCbVz6sMXCZvlrAdmNNKapK21Jd7hsaFS832wEZd7rFOqvD35HW1iQ9NFuBZX
				GEUeMhQoL+7+tHrXTJYONrpndUQfNJwfGYSR7AWhzzAOTwzOXaVuabvS5hhYrn07
				pntfrDtAVYHjh4O+joQJV1sYwI/F1wPZzMvU1hMh12RaoYYNsZFBJzszi0E+x6Cu
				GjLKjkbGW08uAk3/k4daV6BesaqewVIlOCxxyqdDe0xIUusJMKEX17C/jlvIJoH8
				A/xeMEdNl6rqc2h5Ge6EPClMKvkhMzmOJvbecTLKP6gJ7IAnGV+lRxgHuIxmwtCN
				SSmnWHkcE8C70A==
				=ynen
				-----END PGP PUBLIC KEY BLOCK-----
			EOF
			;;&
		rhel)
			gpg_file="/etc/pki/rpm-gpg/RPM-GPG-KEY-apprepo"
			cat > /etc/yum.repos.d/vultr-apprepo.repo <<- EOF
				[apprepo]
				name=apprepo
				baseurl=https://apprepo.vultr.com/rhel/
				enabled=1
				gpgcheck=1
				gpgkey=file://${gpg_file}
			EOF
			mv "${temp_file}" "${gpg_file}"
			;;
		deb)
			case "${OCTENIUM_COMMON_OSID}" in
				debian*) os_name="debian" ;;
				ubuntu*) os_name="ubuntu" ;;
				*) invalid_os_selection ;;
			esac

			install_packages gnupg2
			gpg_file="/etc/apt/trusted.gpg.d/vultr-apprepo.gpg"

			if [[ "${arch}" == x86_64 ]]; then
				arch="amd64"
			fi

			printf 'deb [arch=%s signed-by=%s] %s %s %s\n' \
				"${arch}" \
				"${gpg_file}" \
				"https://apprepo.vultr.com/${os_name}" \
				"universal" \
				"main" > /etc/apt/sources.list.d/vultr-apprepo.list
			gpg --dearmor < "${temp_file}" > "${gpg_file}"

			rm -f "${temp_file}"
			;;
		*) : "GPG key and repo configuration not implemented for this OS/distribution." ;;
	esac
}

OCTENIUM_COMMON_ARCHITECTURE="$(
	set -euo pipefail
	get_architecture
)"

OCTENIUM_COMMON_OSID="$(
	set -euo pipefail
	get_os
)"

OCTENIUM_COMMON_FLAVOR="$(
	set -euo pipefail
	get_flavor
)"

export \
	NVIDIA_DRIVER_VERSION \
	OCTENIUM_COMMON_ARCHITECTURE \
	OCTENIUM_COMMON_FLAVOR \
	OCTENIUM_COMMON_OSID
