#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Setting variables"
########
export APP_SHORT="drupal"
cockpit_port="9080"
php_version="8.3"
metadata_variable_list=(
	"dbname"
	"dbuser"
	"dbpass"
	"installuser"
	"installpass"
	"pmadbuser"
	"pmadbpass"
	"pmamodalpass"
)

########
echo "Retrieve necessary variables from metadata"
########

ip="$(curl -s https://files.oct-backend.com/octeniumvz/ip.php)"

for var in "${metadata_variable_list[@]}"; do
	get_var "${var}"
done

# Exit if any required variables are not set
: "${ip:?}" \
	"${dbname:?}" "${dbuser:?}" "${dbpass:?}" \
	"${installuser:?}" "${installpass:?}" \
	"${pmadbpass:?}" "${pmadbuser:?}" "${pmamodalpass:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure SQL Database"
########
install_mysql "${mysql_ibps:-"0"}" "${mysql_toc:-"0"}"
systemctl enable --now mariadb.service

########
echo "Install and configure PHP"
########
install_php "${php_version}" "optimize"

########
echo "Install and configure nginx"
########
install_nginx

########
echo "Install 'certbot'"
########
install_certbot

########
echo "Install, configure, and secure PHPMyAdmin"
########

install_phpmyadmin "pmauser${pmadbuser}" "${pmadbpass}"
htpasswd -B -i -c /etc/nginx/htpasswd/phpmyadmin "pmauser${pmadbuser}" <<< "${pmamodalpass}" > /dev/null 2>&1

########
echo "Generate TLS certificates"
########
generate_self_signed_cert

########
echo "Install and configure Cockpit"
########
install_cockpit "${ip}" "${cockpit_port}"

########
echo "Install and configure Drupal"
########
mkdir -p /etc/nginx/orig/
mv drupal_http* /etc/nginx/orig/
cp /etc/nginx/orig/drupal_http{,s}.conf /etc/nginx/sites-available/
ln -s /etc/nginx/sites-{available,enabled}/drupal_http.conf
ln -s /etc/nginx/sites-{available,enabled}/drupal_https.conf

(
	cd /var/www/html
	curl --fail --silent --show-error --location --output drupal.zip \
		https://www.drupal.org/download-latest/zip

	unzip -qq drupal.zip
	rm -f drupal.zip

	src_dir="$(find /var/www/html -maxdepth 1 -type d -name 'drupal-*')"
	mv "${src_dir}" drupal
	mv drupal/* drupal/.htaccess .
	rm -rf drupal
)
chown -R www-data:www-data /var/www/html

########
echo "Configure Drupal Database, Drupal default configuration, and secure folders"
########
mysql --defaults-file=/root/.my.cnf -u root -e \
	"CREATE DATABASE dr${dbname} CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci;"
mysql --defaults-file=/root/.my.cnf -u root -e \
	"CREATE USER 'drupal${dbuser}'@'localhost' IDENTIFIED BY '${dbpass}';"
mysql --defaults-file=/root/.my.cnf -u root -e \
	"GRANT ALL PRIVILEGES ON dr${dbname}.* TO 'drupal${dbuser}'@'localhost'; FLUSH PRIVILEGES;"

# NOTE: Changing the default.settings.php will brick the Drupal install if the user re-inputs the database
# settings during the install wizard.
chmod -w /var/www/html/sites/default/default.{services.yml,settings.php}

htpasswd -B -i -c /etc/nginx/htpasswd/install "user${installuser}" <<< "${installpass}" > /dev/null 2>&1

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"

########
echo "Enable services"
########
systemctl enable --now \
	cockpit.socket \
	"php${php_version}-fpm.service" \
	nginx.service

cleanup_imageless