#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Setting variables"
########
export APP_SHORT="joomla"
cockpit_port="9080"
php_version="8.2"
metadata_variable_list=(
	"installuser"
	"installpass"
	"dbname"
	"dbuser"
	"dbpass"
	"pmadbuser"
	"pmadbpass"
	"pmamodalpass"
)

########
echo "Retrieve necessary variables from metadata"
########
get_ip

for var in "${metadata_variable_list[@]}"; do
	get_var "${var}"
done

# Exit if any required variables are not set
: "${ip:?}" \
	"${dbname:?}" "${dbuser:?}" "${dbpass:?}" \
	"${installuser:?}" "${installpass:?}" \
	"${pmadbpass:?}" "${pmadbuser:?}" "${pmamodalpass:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure SQL Database"
########
install_mysql "${mysql_ibps:-"0"}" "${mysql_toc:-"0"}"
systemctl enable --now mariadb.service

########
echo "Install and configure PHP"
########
install_php "${php_version}" "optimize"
printf '%s\n' "php_admin_flag[output_buffering] = off" >> "/etc/php/${php_version}/fpm/pool.d/www.conf"

########
echo "Install and configure nginx"
########
install_nginx

########
echo "Install 'certbot'"
########
install_certbot

########
echo "Install, configure, and secure PHPMyAdmin"
########
install_phpmyadmin "pmauser${pmadbuser}" "${pmadbpass}"
htpasswd -B -i -c /etc/nginx/htpasswd/phpmyadmin "pmauser${pmadbuser}" <<< "${pmamodalpass}" > /dev/null 2>&1

########
echo "Generate TLS certificates"
########
generate_self_signed_cert

########
echo "Install and configure Cockpit"
########
install_cockpit "${ip}" "${cockpit_port}"

########
echo "Install and configure Joomla"
########
mkdir -p /etc/nginx/orig/
mv joomla_http* /etc/nginx/orig/
cp /etc/nginx/orig/joomla_http{,s}.conf /etc/nginx/sites-available/
ln -s /etc/nginx/sites-{available,enabled}/joomla_http.conf
ln -s /etc/nginx/sites-{available,enabled}/joomla_https.conf

(
	cd /var/www/html
	url_base="https://downloads.joomla.org/us"
	url="${url_base}/$(
		set -euo pipefail
		curl --fail --silent --show-error --location "${url_base}" |
			grep --only-matching "cms/joomla.*/Joomla_.*-Stable-Full_Package\.zip?format=zip"
	)"

	curl --fail --silent --show-error --location --output latest.zip "${url}"
	unzip -qq latest.zip
	rm -f latest.zip
)
chown -R www-data:www-data /var/www/html

########
echo "Configure Joomla Database and secure folders"
########
mysql --defaults-file=/root/.my.cnf -u root -e \
	"CREATE DATABASE jdb${dbname} CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci;"
mysql --defaults-file=/root/.my.cnf -u root -e \
	"CREATE USER 'juser${dbuser}'@'localhost' IDENTIFIED BY '${dbpass}';"
mysql --defaults-file=/root/.my.cnf -u root -e \
	"GRANT ALL PRIVILEGES ON jdb${dbname}.* TO 'juser${dbuser}'@'localhost'; FLUSH PRIVILEGES;"

htpasswd -B -i -c /etc/nginx/htpasswd/install "user${installuser}" <<< "${installpass}" > /dev/null 2>&1

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"

########
echo "Enable services"
########
systemctl enable --now \
	cockpit.socket \
	"php${php_version}-fpm.service" \
	nginx.service

########
echo "Cleanup the environment"
########
cleanup_imageless
