#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Set variables"
########
DEFAULT_DB="odoo"
ODOO_CONF="/etc/odoo/odoo.conf"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Installing NGINX"
########
install_nginx
mv odoo.conf /etc/nginx/sites-available/
ln -s /etc/nginx/sites-available/odoo.conf /etc/nginx/sites-enabled/odoo.conf

########
echo "Install 'certbot'"
########
install_certbot

########
echo "Generate TLS certificates"
########
generate_self_signed_cert

########
echo "Install and configure PostgreSQL Database"
########
install_packages postgresql

# Create PostgreSQL user and database as the postgres user
runuser -l postgres -c "createuser -s odoo"
runuser -l postgres -c "psql -c 'CREATE DATABASE ${DEFAULT_DB} OWNER odoo;'"

########
echo "Install Odoo"
########
curl -fsSL https://nightly.odoo.com/odoo.key | gpg --dearmor -o /usr/share/keyrings/odoo-archive-keyring.gpg
printf 'deb [signed-by=/usr/share/keyrings/odoo-archive-keyring.gpg] https://nightly.odoo.com/master/nightly/deb/ ./' > /etc/apt/sources.list.d/odoo.list
install_packages odoo

# Ensure the database connection parameters are correctly set
sed -i -e '/^db_host/d' -e '/^db_port/d' -e '/^db_user/d' -e '/^db_password/d' "${ODOO_CONF}"
# Run odoo as the odoo user
runuser -l odoo -s /bin/bash -c "odoo -c /etc/odoo/odoo.conf -d ${DEFAULT_DB} -i base --stop-after-init"

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"
cleanup_imageless

########
echo "Enabling and restarting Odoo service..."
########
systemctl enable --now nginx odoo
########
echo "Configure firewall"
########
ufw default deny incoming
ufw allow 80/tcp
ufw allow 443/tcp
ufw allow 8069/tcp
ufw status verbose
