#!/bin/bash

set -euo pipefail
shopt -s inherit_errexit

if [[ -f /var/lib/oct/.application-installed ]]; then
	echo "Application script completion detected - not re-running"
	exit 0
fi

. /opt/oct-installer/common.sh

########
echo "Retrieve necessary metadata values"
########
for var in "desktopuser" "desktoppassword" "vncpassword"; do
	get_var "${var}"
done

: "${desktopuser:?}" "${desktoppassword:?}" "${vncpassword:?}"

########
echo "Disable unattended upgrades while configuring app"
########
toggle_unattended_upgrades "disable"

########
echo "Install and configure Desktop Environment"
########
install_packages \
	firefox \
	sddm \
	sssd \
	sssd-tools \
	ubuntu-gnome-desktop \
	x11vnc

systemctl disable --now gdm3.service
echo "/usr/bin/sddm" > /etc/X11/default-display-manager

########
echo "Create user account"
########
useradd \
	--skel /etc/skel \
	--create-home \
	--home-dir "/home/${desktopuser}" \
	--shell /bin/bash \
	--groups sudo \
	"${desktopuser}"

chpasswd <<< "${desktopuser}:${desktoppassword}"

########
echo "Configure VNC server"
########
mkdir -p "/home/${desktopuser}/.vnc"
x11vnc -storepasswd "${vncpassword}" "/home/${desktopuser}/.vnc/passwd"
chmod 600 "/home/${desktopuser}/.vnc/passwd"
chown -R "${desktopuser}:${desktopuser}" "/home/${desktopuser}/.vnc"

sed -e "s/{{DESKTOP_USER}}/${desktopuser}/" \
	/opt/oct-installer/octenium-x11vnc.service > /usr/lib/systemd/system/x11vnc.service

cat > /etc/sddm.conf << EOF
[Users]
HideUsers=docker,ubuntu
EOF

if command -v nvidia-xconfig > /dev/null 2>&1; then
	nvidia-xconfig
fi

# Install fbdev if BM
install_fbdev

########
echo "Remove any remaining unnecessary packages"
########
autoremove_packages

########
echo "Re-enable unattended upgrades as app is configured"
########
toggle_unattended_upgrades "enable"

########
echo "Enable services"
########
systemctl stop rc-local.service
systemctl enable \
    sddm.service \
    sssd.service \
    x11vnc.service

########
echo "Cleanup the environment"
########
cleanup_imageless
